<?php
/**
 * Plugin Name: cAPItula WP Campaign Configuration
 * Plugin URI: https://capitula.co.uk
 * Description: By invitation only. For registered Affiliates. Automatically integrates Capitula affiliate tracking system with WordPress and WooCommerce. Handles CDN injection, conversion tracking, and cross-domain compatibility.
 * Version: 1.10.30
 * Author: Capitula Team
 * Author URI: https://capitula.co.uk
 * License: GPL v2 or later
 * Text Domain: capitula-campaign-config
 * Domain Path: /languages
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('CAPITULA_PLUGIN_VERSION', '1.10.30');
define('CAPITULA_PLUGIN_URL', plugin_dir_url(__FILE__));
define('CAPITULA_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('CAPITULA_PLUGIN_BASENAME', plugin_basename(__FILE__));
define('CAPITULA_UPDATE_SERVER', 'https://capitula.co.uk/downloads/update-server-api.php');

/**
 * Main Capitula Campaign Configuration Class
 */
class CapitulaCampaignConfig {
    
    private static $instance = null;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        add_action('init', array($this, 'init'));
        add_action('init', array($this, 'handle_capitula_token')); // Handle token early
        add_action('wp_head', array($this, 'inject_cdn_scripts'));
        
        // Initialize auto-updater
        if (is_admin()) {
            new CapitulaPluginUpdater(CAPITULA_PLUGIN_BASENAME, CAPITULA_PLUGIN_VERSION, CAPITULA_UPDATE_SERVER);
        }
        
        // Admin hooks
        if (is_admin()) {
            add_action('admin_menu', array($this, 'add_admin_menu'));
            add_action('admin_init', array($this, 'register_settings'));
            add_action('admin_head', array($this, 'admin_custom_css'));
        }
        
        // Plugin activation/deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    /**
     * Initialize plugin
     */
    public function init() {
        // Load text domain for translations
        load_plugin_textdomain('capitula-campaign-config', false, dirname(CAPITULA_PLUGIN_BASENAME) . '/languages');
        
        // Initialize WooCommerce integration if available and enabled
        if (class_exists('WooCommerce') && get_option('capitula_enable_woocommerce', 'yes') === 'yes') {
            // Use multiple hooks for better compatibility
            add_action('woocommerce_thankyou', array($this, 'woocommerce_conversion_tracking'));
            add_action('woocommerce_order_details_after_order_table', array($this, 'woocommerce_conversion_tracking'));
        }
        
        // Initialize user registration tracking
        add_action('wp_login', array($this, 'user_login_tracking'), 10, 2);
        add_action('user_register', array($this, 'user_registration_tracking'));
        
        // Initialize contact form integrations
        $this->init_contact_form_integrations();
        
        // Add action to output stored form emails
        add_action('wp_footer', array($this, 'output_stored_form_emails'));
    }
    
    /**
     * Inject CDN scripts into header
     */
    public function inject_cdn_scripts() {
        $cdn_url = get_option('capitula_cdn_url', 'https://api.capitula.co.uk');
        $enable_fallback = get_option('capitula_enable_fallback', 'no');
        
        if ($enable_fallback === 'yes') {
            // Cross-domain safe method WITHOUT document.write to eliminate browser warnings
            ?>
            <script type="text/javascript">
            (function() {
                // Load Capitula CDN script dynamically with CORS handling
                var capitulaScript = document.createElement('script');
                capitulaScript.type = 'text/javascript';
                capitulaScript.src = '<?php echo esc_js($cdn_url); ?>/assets/js/capitula_cdn.js?v=' + Date.now() + '&cb=<?php echo time(); ?>';
                capitulaScript.defer = true;
                capitulaScript.crossOrigin = 'anonymous';
                
                // Add error handler for CORS issues
                capitulaScript.onerror = function() {
                    console.log('Capitula: CORS error detected, using fallback method');
                    // Fallback: Load script content via WordPress AJAX proxy
                    fetch('<?php echo admin_url('admin-ajax.php'); ?>?action=capitula_proxy_cdn&v=' + Date.now())
                        .then(function(response) { return response.text(); })
                        .then(function(jsCode) {
                            var fallbackScript = document.createElement('script');
                            fallbackScript.textContent = jsCode;
                            document.head.appendChild(fallbackScript);
                            console.log('Capitula: Fallback CDN loaded successfully');
                        })
                        .catch(function(error) {
                            console.log('Capitula: Fallback CDN failed:', error);
                        });
                };
                
                document.head.appendChild(capitulaScript);
                
                // Load jQuery if not available
                if (typeof jQuery === 'undefined') {
                    var jqueryScript = document.createElement('script');
                    jqueryScript.type = 'text/javascript';
                    jqueryScript.src = 'https://ajax.googleapis.com/ajax/libs/jquery/3.7.1/jquery.min.js';
                    jqueryScript.defer = true;
                    jqueryScript.crossOrigin = 'anonymous';
                    document.head.appendChild(jqueryScript);
                }
            })();
            </script>
            <?php
        } else {
            // Original Capitula code exactly as provided
            ?>
            <script>document.write("<script type='text/javascript' src='<?php echo esc_js($cdn_url); ?>/assets/js/capitula_cdn.js?v=" + Date.now() + "&cb=<?php echo time(); ?>'><\/script>");</script>
            <script type="text/javascript">if(typeof jQuery == 'undefined'){document.write('<script type="text/javascript" src="https://ajax.googleapis.com/ajax/libs/jquery/3.7.1/jquery.min.js"></'+'script>');}</script>
            <?php
        }
    }
    
    /**
     * Handle capitula_token URL parameter and save as cookie
     */
    public function handle_capitula_token() {
        // Check if capitula_token is in the URL
        if (isset($_GET['capitula_token']) && !empty($_GET['capitula_token'])) {
            $token = sanitize_text_field($_GET['capitula_token']);
            
            // Parse token - handle both old (concat) and new (colon-separated) formats
            $ad_position = '';
            $campaign_token = '';
            
            if (strpos($token, ':') !== false) {
                // New format: "position:token"
                $parts = explode(':', $token, 2);
                $ad_position = $parts[0];
                $campaign_token = $parts[1];
            } else {
                // Legacy format: first char = ad_position, rest = campaign_token
                if (strlen($token) >= 2) {
                    $ad_position = substr($token, 0, 1);    // First character
                    $campaign_token = substr($token, 1);     // Rest of string
                } else {
                    // Invalid token format
                    return;
                }
            }
            
            // Set cookie for 30 days (secure, SameSite=Lax)
            $cookie_value = $ad_position . ':' . $campaign_token;
            $expire_time = time() + (30 * 24 * 60 * 60); // 30 days
            
            // Set cookie (compatible with JavaScript reading)
            setcookie('uac_token', $cookie_value, $expire_time, '/', '', is_ssl(), false);
            
            // Also set it for immediate use in current request
            $_COOKIE['uac_token'] = $cookie_value;
            
            // Debug log
            error_log('Capitula: Token saved - Position: ' . $ad_position . ', Campaign: ' . $campaign_token);
        }
    }
    
    /**
     * WooCommerce conversion tracking
     * Enhanced to match previous functionality and use order number
     */
    public function woocommerce_conversion_tracking($order_id) {
        if (!$order_id) return;
        
        // Check if WooCommerce integration is enabled
        if (get_option('capitula_enable_woocommerce', 'yes') !== 'yes') {
            return;
        }
        
        $order = wc_get_order($order_id);
        if (!$order) return;
        
        // Use order number instead of ID (matches previous functionality)
        $order_number = $order->get_order_number();
        $total_price = $order->get_total();
        
        // Output hidden Capitula tags for tracking (exactly like previous function)
        echo "<div style='display: none;'>";
        echo "<capitula_trans>{$order_number}</capitula_trans>";
        echo "<capitula_pay>{$total_price}</capitula_pay>";
        echo "</div>";
        
        // Add JavaScript to handle the conversion tracking
        ?>
        <script type="text/javascript">
        document.addEventListener('DOMContentLoaded', function() {
            // Ensure Capitula CDN is loaded before processing conversions
            if (typeof window.capitulaProcessConversion === 'function') {
                window.capitulaProcessConversion();
            } else {
                // Retry after a short delay
                setTimeout(function() {
                    if (typeof window.capitulaProcessConversion === 'function') {
                        window.capitulaProcessConversion();
                    }
                }, 1000);
            }
        });
        </script>
        <?php
    }
    
    /**
     * User registration tracking
     */
    public function user_registration_tracking($user_id) {
        $user = get_userdata($user_id);
        if ($user && $user->user_email) {
            // Store email for thank you page display
            set_transient('capitula_new_user_email_' . session_id(), $user->user_email, 300); // 5 minutes
        }
    }
    
    /**
     * User login tracking (for registration confirmation)
     */
    public function user_login_tracking($user_login, $user) {
        // Check if this is a new user login (within registration flow)
        $new_user_email = get_transient('capitula_new_user_email_' . session_id());
        if ($new_user_email && $user->user_email === $new_user_email) {
            echo '<div style="display:none;">';
            echo '<capitula_email>' . esc_html($user->user_email) . '</capitula_email>';
            echo '</div>';
            
            // Clean up transient
            delete_transient('capitula_new_user_email_' . session_id());
        }
    }
    
    /**
     * Initialize contact form integrations
     */
    private function init_contact_form_integrations() {
        $tracking_method = get_option('capitula_cpl_tracking_method', 'simple_ajax');
        
        switch ($tracking_method) {
            case 'pure_php':
                $this->init_pure_php_tracking();
                break;
            case 'webhook_api':
                $this->init_webhook_api_tracking();
                break;
            case 'database_queue':
                $this->init_database_queue_tracking();
                break;
            case 'simple_ajax':
            default:
                $this->init_simple_ajax_tracking();
                break;
        }
        
        // Always keep basic fallback hooks
        if (defined('WPCF7_VERSION')) {
            add_action('wpcf7_mail_sent', array($this, 'cf7_conversion_tracking'));
        }
        
        if (class_exists('GFForms')) {
            add_action('gform_after_submission', array($this, 'gravity_forms_tracking'), 10, 2);
        }
        
        if (class_exists('Ninja_Forms')) {
            add_action('ninja_forms_after_submission', array($this, 'ninja_forms_tracking'));
        }
        
        // Add universal AJAX handler
        add_action('wp_ajax_capitula_universal_cpl', array($this, 'ajax_universal_cpl'));
        add_action('wp_ajax_nopriv_capitula_universal_cpl', array($this, 'ajax_universal_cpl'));
        
        // Add CORS proxy handler for CDN script
        add_action('wp_ajax_capitula_proxy_cdn', array($this, 'ajax_proxy_cdn'));
        add_action('wp_ajax_nopriv_capitula_proxy_cdn', array($this, 'ajax_proxy_cdn'));
    }
    
    /**
     * Initialize Simple AJAX Tracking (Current Method)
     */
    private function init_simple_ajax_tracking() {
        add_action('wp_footer', array($this, 'simple_server_side_cpl_tracking'));
        // Using unified AJAX handler: capitula_universal_cpl
        
        // Also initialize Universal tracking if enabled (for WPForms support)
        $universal_enabled = get_option('capitula_universal_tracking_enabled', 'yes');
        if ($universal_enabled === 'yes') {
            $this->init_universal_form_tracking();
        }
    }
    
    /**
     * Initialize Pure PHP Tracking (Universal Form Detection)
     */
    private function init_pure_php_tracking() {
        // Universal form tracking - works with ANY form plugin (if enabled)
        $universal_enabled = get_option('capitula_universal_tracking_enabled', 'yes');
        if ($universal_enabled === 'yes') {
            $this->init_universal_form_tracking();
        }
        
        // Still keep specific plugin hooks for better integration (as fallback)
        $configured_plugins = get_option('capitula_configured_form_plugins', array());
        
        // Ensure it's an array (fix for when option is saved as string)
        if (!is_array($configured_plugins)) {
            $configured_plugins = array();
        }
        
        // Only add plugin-specific hooks if user has configured them
        if (in_array('contact_form_7', $configured_plugins) && defined('WPCF7_VERSION')) {
            add_action('wpcf7_mail_sent', array($this, 'cf7_pure_php_cpl_tracking'), 20);
        }
        
        if (in_array('gravity_forms', $configured_plugins) && class_exists('GFForms')) {
            add_action('gform_after_submission', array($this, 'gf_pure_php_cpl_tracking'), 20, 2);
        }
        
        if (in_array('ninja_forms', $configured_plugins) && class_exists('Ninja_Forms')) {
            add_action('ninja_forms_after_submission', array($this, 'nf_pure_php_cpl_tracking'), 20);
        }
        
        if (in_array('wpforms', $configured_plugins) && function_exists('wpforms')) {
            add_action('wpforms_process_complete', array($this, 'wpforms_pure_php_cpl_tracking'), 20, 4);
        }
        
        if (in_array('fluent_forms', $configured_plugins) && class_exists('FluentForm\Framework\Foundation\Application')) {
            add_action('fluentform_submission_inserted', array($this, 'fluentforms_pure_php_cpl_tracking'), 20, 3);
        }
        
        if (in_array('jotform', $configured_plugins)) {
            // JotForm uses webhooks - we'll handle this via universal detection
            add_action('init', array($this, 'jotform_webhook_listener'));
        }
    }
    
    /**
     * Initialize Webhook/API Tracking
     */
    private function init_webhook_api_tracking() {
        add_action('rest_api_init', array($this, 'register_cpl_webhook_routes'));
        add_action('wp_footer', array($this, 'webhook_api_tracking_javascript'));
    }
    
    /**
     * Initialize Database Queue Tracking
     */
    private function init_database_queue_tracking() {
        add_action('wp_footer', array($this, 'database_queue_tracking_javascript'));
        add_action('wp_ajax_capitula_queue_cpl', array($this, 'ajax_queue_cpl'));
        add_action('wp_ajax_nopriv_capitula_queue_cpl', array($this, 'ajax_queue_cpl'));
        
        // Schedule cron job for processing queue
        if (!wp_next_scheduled('capitula_process_cpl_queue')) {
            wp_schedule_event(time(), 'every_minute', 'capitula_process_cpl_queue');
        }
        add_action('capitula_process_cpl_queue', array($this, 'process_cpl_queue'));
    }
    
    /**
     * Contact Form 7 conversion tracking
     */
    public function cf7_conversion_tracking($contact_form) {
        $submission = WPCF7_Submission::get_instance();
        if ($submission) {
            $posted_data = $submission->get_posted_data();
            if (isset($posted_data['your-email'])) {
                $email = sanitize_email($posted_data['your-email']);
                if ($email) {
                    // Store email for display in thank you message
                    set_transient('capitula_cf7_email_' . session_id(), $email, 300); // 5 minutes
                }
            }
        }
    }
    
    /**
     * Contact Form 7 AJAX response (no modifications needed)
     */
    public function modify_cf7_response($response, $result) {
        // CPL tracking is handled directly via JavaScript, no response modification needed
        return $response;
    }
    
    /**
     * Simple server-side CPL tracking - triggers after successful form submission
     * DISABLED when universal tracking is enabled to prevent duplicate CPLs
     */
    public function simple_server_side_cpl_tracking() {
        // Check if universal tracking is enabled
        $universal_enabled = get_option('capitula_universal_tracking_enabled', 'yes');
        
        // If universal tracking is enabled, PHP hooks handle all forms - no need for JavaScript
        if ($universal_enabled === 'yes') {
            ?>
            <script type="text/javascript">
            console.log('Capitula: Universal tracking enabled - JavaScript form tracking disabled (PHP hooks active)');
            </script>
            <?php
            return; // Exit - no JavaScript tracking needed
        }
        
        // Only output JavaScript tracking if universal tracking is DISABLED
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        
        if (!$token) {
            return;
        }
        
        ?>
        <script type="text/javascript">
        // Simple approach: Just wait for forms to be submitted successfully
        document.addEventListener('DOMContentLoaded', function() {
            console.log('Capitula: Simple CPL tracking initialized (JavaScript mode)');
            
            // Contact Form 7 success
            document.addEventListener('wpcf7mailsent', function(event) {
                console.log('Capitula: CF7 form submitted successfully');
                handleSuccessfulSubmission(event.target);
            });
            
            // Gravity Forms success  
            document.addEventListener('gform_confirmation_loaded', function(event) {
                console.log('Capitula: Gravity Forms submitted successfully');
                var forms = document.querySelectorAll('form[id*="gform"]');
                if (forms.length > 0) {
                    handleSuccessfulSubmission(forms[0]);
                }
            });
            
            function handleSuccessfulSubmission(form) {
                console.log('Capitula: Processing successful form submission');
                
                // Find email field in the submitted form
                var emailField = form.querySelector(
                    'input[name="your-email"], ' +
                    'input[name="email"], ' +
                    'input[type="email"], ' +
                    'input[name*="email"]'
                );
                
                if (emailField && emailField.value && emailField.value.includes('@')) {
                    var email = emailField.value.trim();
                    console.log('Capitula: Found email in successful form:', email);
                    
                    // Trigger server-side CPL tracking via AJAX
                    triggerServerSideCPL(email);
                } else {
                    console.log('Capitula: No email found in successful form');
                }
            }
            
            function triggerServerSideCPL(email) {
                console.log('Capitula: Triggering server-side CPL for:', email);
                
                // Small delay to give PHP hooks priority over JavaScript
                setTimeout(function() {
                    console.log('Capitula: Processing JavaScript CPL (after PHP hook delay)');
                    
                    // Send email to our WordPress endpoint for server-side processing
                    fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'action=capitula_universal_cpl&email=' + encodeURIComponent(email) + '&token=' + encodeURIComponent('<?php echo esc_js($token); ?>')
                })
                .then(response => response.json())
                .then(data => {
                    console.log('Capitula: Server-side CPL response:', data);
                })
                .catch(error => {
                    console.log('Capitula: Server-side CPL error:', error);
                });
                
                }, 100); // 100ms delay to give PHP hooks priority
            }
        });
        </script>
        <?php
    }
    
    // Old ajax_process_cpl method removed - now using unified ajax_universal_cpl
    
    /**
     * AJAX CDN Proxy Handler - Bypasses CORS issues
     */
    public function ajax_proxy_cdn() {
        $cdn_url = get_option('capitula_cdn_url', 'https://api.capitula.co.uk');
        $script_url = $cdn_url . '/assets/js/capitula_cdn.js';
        
        // Fetch CDN script content
        $response = wp_remote_get($script_url, [
            'timeout' => 30,
            'headers' => [
                'User-Agent' => 'WordPress Capitula Plugin'
            ]
        ]);
        
        if (is_wp_error($response)) {
            header('Content-Type: application/javascript');
            echo '// Capitula CDN proxy error: ' . $response->get_error_message();
            echo "\nconsole.log('Capitula: CDN proxy failed - " . esc_js($response->get_error_message()) . "');";
            wp_die();
        }
        
        $script_content = wp_remote_retrieve_body($response);
        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code === 200 && !empty($script_content)) {
            header('Content-Type: application/javascript');
            header('Cache-Control: public, max-age=3600'); // Cache for 1 hour
            echo $script_content;
        } else {
            header('Content-Type: application/javascript');
            echo '// Capitula CDN proxy error: HTTP ' . $response_code;
            echo "\nconsole.log('Capitula: CDN proxy failed - HTTP " . $response_code . "');";
        }
        
        wp_die();
    }
    
    
    /**
     * Get user's real IP address
     */
    private function get_user_ip() {
        $ip_keys = ['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR'];
        
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                foreach (explode(',', $_SERVER[$key]) as $ip) {
                    $ip = trim($ip);
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                        return $ip;
                    }
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
    }
    
    /**
     * Get device type from user agent
     */
    private function get_device_type() {
        $user_agent = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');
        
        if (preg_match('/mobile|android|iphone|ipod|blackberry|iemobile|opera mini/i', $user_agent)) {
            return 'Mobile';
        } elseif (preg_match('/tablet|ipad|playbook|silk/i', $user_agent)) {
            return 'Tablet';
        } else {
            return 'Desktop';
        }
    }
    
    /**
     * Check if current request is WooCommerce-related
     * Prevents CPL tracking during WooCommerce checkout/order process (only CPA should fire)
     */
    private function is_woocommerce_request() {
        // Check if WooCommerce is even installed
        if (!class_exists('WooCommerce')) {
            return false;
        }
        
        // Check if WooCommerce integration is enabled in plugin settings
        if (get_option('capitula_enable_woocommerce', 'yes') !== 'yes') {
            return false; // If WooCommerce tracking disabled, allow CPL tracking
        }
        
        // Check for checkout page
        if (function_exists('is_checkout') && is_checkout()) {
            return true;
        }
        
        // Check for order received/thank you page
        if (function_exists('is_order_received_page') && is_order_received_page()) {
            return true;
        }
        
        // Check for WooCommerce AJAX actions in POST data
        if (isset($_POST['action']) && 
            (strpos($_POST['action'], 'woocommerce') !== false || 
             strpos($_POST['action'], 'wc_') !== false)) {
            return true;
        }
        
        // Check for WooCommerce checkout form submission
        if (isset($_POST['woocommerce_checkout_place_order']) || 
            isset($_POST['woocommerce-process-checkout-nonce']) ||
            isset($_POST['payment_method']) ||
            isset($_POST['billing_email'])) { // billing_email indicates WooCommerce checkout
            return true;
        }
        
        // Check if we're in WooCommerce order processing
        if (did_action('woocommerce_checkout_process') || 
            did_action('woocommerce_checkout_order_processed')) {
            return true;
        }
        
        // Check URL for WooCommerce endpoints
        $request_uri = $_SERVER['REQUEST_URI'] ?? '';
        if (strpos($request_uri, '/checkout') !== false || 
            strpos($request_uri, '/order-received') !== false ||
            strpos($request_uri, 'wc-ajax') !== false) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Contact Form 7 Pure PHP CPL Tracking
     */
    public function cf7_pure_php_cpl_tracking($contact_form) {
        $submission = WPCF7_Submission::get_instance();
        if ($submission) {
            $posted_data = $submission->get_posted_data();
            if (isset($posted_data['your-email'])) {
                $email = sanitize_email($posted_data['your-email']);
                if ($email) {
                    $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
                    if ($token) {
                        $this->send_pure_php_cpl_tracking($email, $token);
                    }
                }
            }
        }
    }
    
    /**
     * Gravity Forms Pure PHP CPL Tracking
     */
    public function gf_pure_php_cpl_tracking($entry, $form) {
        foreach ($form['fields'] as $field) {
            if ($field->type === 'email' && !empty($entry[$field->id])) {
                $email = sanitize_email($entry[$field->id]);
                if ($email) {
                    $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
                    if ($token) {
                        $this->send_pure_php_cpl_tracking($email, $token);
                    }
                    break;
                }
            }
        }
    }
    
    /**
     * Ninja Forms Pure PHP CPL Tracking
     */
    public function nf_pure_php_cpl_tracking($form_data) {
        if (isset($form_data['fields'])) {
            foreach ($form_data['fields'] as $field) {
                if ($field['type'] === 'email' && !empty($field['value'])) {
                    $email = sanitize_email($field['value']);
                    if ($email) {
                        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
                        if ($token) {
                            $this->send_pure_php_cpl_tracking($email, $token);
                        }
                        break;
                    }
                }
            }
        }
    }
    
    /**
     * Initialize Universal Form Detection System
     * Uses multiple detection strategies for ANY form plugin
     */
    private function init_universal_form_tracking() {
        // Strategy 1: Watch REST API requests (wp-json/*)
        add_filter('rest_pre_dispatch', array($this, 'detect_rest_form_submission'), 10, 3);
        
        // Strategy 2: Watch AJAX requests (admin-ajax.php)
        add_action('admin_init', array($this, 'detect_ajax_form_submission'));
        
        // Strategy 3: Catch outgoing emails (wp_mail intercept)
        add_filter('wp_mail', array($this, 'detect_wp_mail_form_submission'), 10, 1);
        
        // Strategy 4: Watch for new posts/custom post types (form submissions stored as posts)
        add_action('save_post', array($this, 'detect_post_form_submission'), 10, 2);
        add_action('transition_post_status', array($this, 'detect_post_status_form_submission'), 10, 3);
        
        // Strategy 5: Universal POST monitoring (existing fallback)
        add_action('wp', array($this, 'universal_post_monitor'), 1);
        
        // Strategy 6: Specific plugin hooks (enhanced)
        $this->init_form_plugin_hooks();
        
        // Internal action for normalized form submissions
        add_action('capitula_form_submitted', array($this, 'handle_normalized_form_submission'), 10, 2);
        
        // AJAX handler for our CPL submissions
        add_action('wp_ajax_capitula_universal_cpl', array($this, 'ajax_universal_cpl'));
        add_action('wp_ajax_nopriv_capitula_universal_cpl', array($this, 'ajax_universal_cpl'));
    }
    
    /**
     * Initialize Form Plugin Hooks
     * Uses native WordPress/plugin hooks for reliable detection
     */
    private function init_form_plugin_hooks() {
        // Contact Form 7
        if (defined('WPCF7_VERSION')) {
            add_action('wpcf7_mail_sent', array($this, 'handle_cf7_submission'), 10, 1);
        }
        
        // WPForms
        if (function_exists('wpforms')) {
            add_action('wpforms_process_complete', array($this, 'handle_wpforms_submission'), 10, 4);
        }
        
        // Gravity Forms
        if (class_exists('GFForms')) {
            add_action('gform_after_submission', array($this, 'handle_gravity_forms_submission'), 10, 2);
        }
        
        // Ninja Forms
        if (class_exists('Ninja_Forms')) {
            add_action('ninja_forms_after_submission', array($this, 'handle_ninja_forms_submission'), 10, 1);
        }
        
        // WSForm - Multiple possible class names and hooks
        if (class_exists('WS_Form_Core') || class_exists('WS_Form') || function_exists('ws_form')) {
            add_action('wsf_submit_post_complete', array($this, 'handle_wsform_submission'), 10, 1);
            add_action('wsf_submit_complete', array($this, 'handle_wsform_submission'), 10, 1);
            add_action('wsf_submit_post', array($this, 'handle_wsform_submission'), 10, 1);
        }
        
        // Elementor Forms
        if (defined('ELEMENTOR_PRO_VERSION')) {
            add_action('elementor_pro/forms/new_record', array($this, 'handle_elementor_forms_submission'), 10, 2);
        }
        
        // Fluent Forms
        if (class_exists('FluentForm\Framework\Foundation\Application')) {
            add_action('fluentform_submission_inserted', array($this, 'handle_fluent_forms_submission'), 10, 3);
        }
    }
    
    /**
     * Universal POST Monitor - Fallback for unknown form plugins
     */
    public function universal_post_monitor() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            return;
        }
        
        // Skip admin, AJAX, and WooCommerce requests (WooCommerce sends CPA, not CPL)
        if (is_admin() || wp_doing_ajax() || $this->is_woocommerce_request()) {
            return;
        }
        
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) {
            return;
        }
        
        // Look for email in POST data
        $email_found = $this->find_email_in_post_data($_POST);
        
        if ($email_found) {
            $this->send_cpl_tracking($email_found, $token, 'POST_monitor');
        }
    }
    
    /**
     * Contact Form 7 Handler
     */
    public function handle_cf7_submission($contact_form) {
        // Skip if WooCommerce checkout (WooCommerce sends CPA, not CPL)
        if ($this->is_woocommerce_request()) return;
        
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) return;
        
        $submission = WPCF7_Submission::get_instance();
        if ($submission) {
            $posted_data = $submission->get_posted_data();
            $email = $this->find_email_in_data($posted_data);
            
            if ($email) {
                $this->send_cpl_tracking($email, $token, 'Contact_Form_7');
            }
        }
    }
    
    /**
     * WPForms Handler
     */
    public function handle_wpforms_submission($fields, $entry, $form_data, $entry_id) {
        // Skip if WooCommerce checkout (WooCommerce sends CPA, not CPL)
        if ($this->is_woocommerce_request()) return;
        
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) return;
        
        foreach ($fields as $field) {
            if (isset($field['type']) && $field['type'] === 'email' && !empty($field['value'])) {
                $email = sanitize_email($field['value']);
                if ($email) {
                    $this->send_cpl_tracking($email, $token, 'WPForms');
                    return;
                }
            }
        }
    }
    
    /**
     * Gravity Forms Handler
     */
    public function handle_gravity_forms_submission($entry, $form) {
        // Skip if WooCommerce checkout (WooCommerce sends CPA, not CPL)
        if ($this->is_woocommerce_request()) return;
        
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) return;
        
        foreach ($form['fields'] as $field) {
            if ($field->type === 'email' && !empty($entry[$field->id])) {
                $email = sanitize_email($entry[$field->id]);
                if ($email) {
                    $this->send_cpl_tracking($email, $token, 'Gravity_Forms');
                    return;
                }
            }
        }
    }
    
    /**
     * Ninja Forms Handler
     */
    public function handle_ninja_forms_submission($form_data) {
        // Skip if WooCommerce checkout (WooCommerce sends CPA, not CPL)
        if ($this->is_woocommerce_request()) return;
        
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) return;
        
        if (isset($form_data['fields'])) {
            foreach ($form_data['fields'] as $field) {
                if ($field['type'] === 'email' && !empty($field['value'])) {
                    $email = sanitize_email($field['value']);
                    if ($email) {
                        $this->send_cpl_tracking($email, $token, 'Ninja_Forms');
                        return;
                    }
                }
            }
        }
    }
    
    /**
     * WSForm Handler
     */
    public function handle_wsform_submission($submit_object) {
        // Skip if WooCommerce checkout (WooCommerce sends CPA, not CPL)
        if ($this->is_woocommerce_request()) return;
        
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) return;
        
        // WSForm can have different data structures
        $email = null;
        
        if (is_object($submit_object)) {
            // Check if it has form data
            if (isset($submit_object->form_object)) {
                $email = $this->find_email_in_wsform_data($submit_object->form_object);
            }
            // Check meta field
            if (!$email && isset($submit_object->meta)) {
                foreach ($submit_object->meta as $field) {
                    if (isset($field->value) && is_email($field->value)) {
                        $email = sanitize_email($field->value);
                        break;
                    }
                }
            }
        } elseif (is_array($submit_object)) {
            $email = $this->find_email_in_data($submit_object);
        }
        
        if ($email) {
            $this->send_cpl_tracking($email, $token, 'WSForm');
        }
    }
    
    /**
     * Elementor Forms Handler
     */
    public function handle_elementor_forms_submission($record, $handler) {
        // Skip if WooCommerce checkout (WooCommerce sends CPA, not CPL)
        if ($this->is_woocommerce_request()) return;
        
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) return;
        
        $fields = $record->get('fields');
        foreach ($fields as $field) {
            if ($field['type'] === 'email' && !empty($field['value'])) {
                $email = sanitize_email($field['value']);
                if ($email) {
                    $this->send_cpl_tracking($email, $token, 'Elementor_Forms');
                    return;
                }
            }
        }
    }
    
    /**
     * Fluent Forms Handler
     */
    public function handle_fluent_forms_submission($entry_id, $form_data, $form) {
        // Skip if WooCommerce checkout (WooCommerce sends CPA, not CPL)
        if ($this->is_woocommerce_request()) return;
        
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) return;
        
        $email = $this->find_email_in_data($form_data);
        
        if ($email) {
            $this->send_cpl_tracking($email, $token, 'Fluent_Forms');
        }
    }
    
    /**
     * Find email in WSForm specific data structure
     */
    private function find_email_in_wsform_data($form_data) {
        if (!$form_data) return null;
        
        // Check various WSForm data structures
        if (isset($form_data->field_types)) {
            foreach ($form_data->field_types as $field) {
                if (isset($field->type) && $field->type === 'email' && !empty($field->value)) {
                    return sanitize_email($field->value);
                }
            }
        }
        
        // Check if it's an array of fields
        if (is_array($form_data)) {
            return $this->find_email_in_data($form_data);
        }
        
        return null;
    }
    
    /**
     * Find email in any data array
     */
    private function find_email_in_data($data) {
        if (!is_array($data)) return null;
        
        foreach ($data as $key => $value) {
            if (is_string($value) && is_email($value)) {
                return sanitize_email($value);
            }
            if (is_array($value)) {
                $nested_email = $this->find_email_in_data($value);
                if ($nested_email) return $nested_email;
            }
        }
        
        return null;
    }
    
    /**
     * Find email in POST data (fallback method)
     */
    private function find_email_in_post_data($post_data) {
        return $this->find_email_in_data($post_data);
    }
    
    /**
     * UNIFIED CPL Tracking with Priority-Based Deduplication
     * Prevents multiple methods from sending CPL for the same submission
     */
    private static $sent_cpls = array();
    private static $cpl_methods_used = array(); // Track which method was used first
    
    private function send_cpl_tracking($email, $token, $source) {
        // Per-request deduplication key - prevents duplicates in same request
        $request_dedup_key = md5($email . $token);
        
        // Check if already sent in THIS request (immediate duplicate prevention)
        if (isset(self::$sent_cpls[$request_dedup_key])) {
            $first_method = self::$cpl_methods_used[$request_dedup_key] ?? 'unknown';
            error_log('Capitula: DUPLICATE CPL BLOCKED - ' . $email . ' from ' . $source . ' (already sent by: ' . $first_method . ' in this request)');
            return false;
        }
        
        // Mark as sent in memory for this request
        self::$sent_cpls[$request_dedup_key] = true;
        self::$cpl_methods_used[$request_dedup_key] = $source;
        
        // Send the CPL
        $result = $this->send_simple_cpl_tracking($email, $token);
        
        error_log('Capitula: ✅ CPL SENT for ' . $email . ' from ' . $source . ' - Result: ' . json_encode($result));
        return true;
    }
    
    /**
     * Minimal JavaScript for Success Message Detection Only (Backup)
     * Minimal backup JavaScript for unknown form plugins only
     */
    public function minimal_success_monitor() {
        // Only add if no specific form plugin hooks are available
        $has_known_plugins = defined('WPCF7_VERSION') || function_exists('wpforms') || 
                           class_exists('GFForms') || class_exists('Ninja_Forms') ||
                           class_exists('WS_Form') || defined('ELEMENTOR_PRO_VERSION');
        
        if ($has_known_plugins) {
            return; // Skip JavaScript if we have proper hooks
        }
        
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) {
            return;
        }
        
        ?>
        <script type="text/javascript">
        // Minimal backup for unknown form plugins only
        console.log('Capitula: Hook-based tracking active, JavaScript backup minimal');
        </script>
        <?php
    }
    
    /**
     * Universal AJAX CPL Handler - Uses Unified Deduplication
     */
    public function ajax_universal_cpl() {
        // Verify request
        if (!isset($_POST['email']) || !isset($_POST['token'])) {
            wp_die(json_encode(['error' => 'Missing parameters']));
        }
        
        $email = sanitize_email($_POST['email']);
        $token = sanitize_text_field($_POST['token']);
        
        if (!$email || !$token) {
            wp_die(json_encode(['error' => 'Invalid parameters']));
        }
        
        // Use unified deduplication system
        $success = $this->send_cpl_tracking($email, $token, 'JavaScript_AJAX');
        
        if ($success) {
            wp_die(json_encode(['success' => true, 'message' => 'CPL sent successfully']));
        } else {
            wp_die(json_encode(['success' => false, 'message' => 'Duplicate CPL prevented - already sent by another method']));
        }
    }
    
    // =================================================================================
    // COMPREHENSIVE UNIVERSAL FORM DETECTION METHODS
    // Multiple strategies to catch ANY form plugin submissions
    // =================================================================================
    
    /**
     * Strategy 1: Detect REST API Form Submissions (wp-json/*)
     * Many modern form plugins use REST endpoints
     */
    public function detect_rest_form_submission($result, $server, $request) {
        // Only process POST requests
        if ($request->get_method() !== 'POST') {
            return $result;
        }
        
        // Skip WooCommerce requests (WooCommerce sends CPA, not CPL)
        if ($this->is_woocommerce_request()) {
            return $result;
        }
        
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) {
            return $result;
        }
        
        $route = $request->get_route();
        $params = $request->get_params();
        
        error_log('Capitula: REST API submission detected - Route: ' . $route);
        
        // Look for email in REST request data
        $email = $this->find_email_in_data($params);
        
        if ($email) {
            error_log('Capitula: Email found in REST API: ' . $email);
            do_action('capitula_form_submitted', $email, 'REST_API_' . $route);
        }
        
        return $result;
    }
    
    /**
     * Strategy 2: Detect AJAX Form Submissions (admin-ajax.php)
     * Monitor AJAX requests for form submissions
     */
    public function detect_ajax_form_submission() {
        if (!defined('DOING_AJAX') || !DOING_AJAX) {
            return;
        }
        
        // Skip WooCommerce requests (WooCommerce sends CPA, not CPL)
        if ($this->is_woocommerce_request()) {
            return;
        }
        
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) {
            return;
        }
        
        $action = isset($_POST['action']) ? $_POST['action'] : '';
        
        // Skip our own actions and WooCommerce actions
        if (strpos($action, 'capitula') !== false || 
            strpos($action, 'woocommerce') !== false ||
            strpos($action, 'wc_') !== false) {
            return;
        }
        
        error_log('Capitula: AJAX submission detected - Action: ' . $action);
        
        // Look for email in AJAX data
        $email = $this->find_email_in_post_data($_POST);
        
        if ($email) {
            error_log('Capitula: Email found in AJAX: ' . $email);
            do_action('capitula_form_submitted', $email, 'AJAX_' . $action);
        }
    }
    
    /**
     * Strategy 3: Detect wp_mail Form Submissions
     * Intercept outgoing emails to capture form submissions
     */
    public function detect_wp_mail_form_submission($args) {
        // Skip WooCommerce requests (WooCommerce sends CPA, not CPL)
        if ($this->is_woocommerce_request()) {
            return $args;
        }
        
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) {
            return $args;
        }
        
        // Check if this looks like a form submission email
        $subject = isset($args['subject']) ? strtolower($args['subject']) : '';
        $message = isset($args['message']) ? strtolower($args['message']) : '';
        
        $form_indicators = ['contact', 'form', 'submission', 'inquiry', 'message', 'feedback', 'quote', 'request'];
        $is_form_email = false;
        
        foreach ($form_indicators as $indicator) {
            if (strpos($subject, $indicator) !== false || strpos($message, $indicator) !== false) {
                $is_form_email = true;
                break;
            }
        }
        
        if ($is_form_email) {
            error_log('Capitula: Form email detected - Subject: ' . $args['subject']);
            
            // Extract email from message content
            $email_pattern = '/[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}/';
            preg_match($email_pattern, $args['message'], $matches);
            
            if (!empty($matches)) {
                $email = $matches[0];
                error_log('Capitula: Email found in wp_mail: ' . $email);
                do_action('capitula_form_submitted', $email, 'WP_MAIL');
            }
        }
        
        return $args;
    }
    
    /**
     * Strategy 4: Detect Form Submissions Saved as Posts
     * Watch for new posts that might be form submissions
     */
    public function detect_post_form_submission($post_id, $post) {
        // Skip WooCommerce requests (WooCommerce sends CPA, not CPL)
        if ($this->is_woocommerce_request()) {
            return;
        }
        
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) {
            return;
        }
        
        // Check if this is a form submission post type
        $form_post_types = ['flamingo_inbound', 'wpcf7_contact_form', 'wpforms_log', 'gf_entry', 'nf_sub'];
        
        if (in_array($post->post_type, $form_post_types)) {
            error_log('Capitula: Form post submission detected - Type: ' . $post->post_type);
            
            // Look for email in post content or meta
            $email = null;
            
            // Check post content
            $email = $this->extract_email_from_text($post->post_content);
            
            // Check post meta if no email found
            if (!$email) {
                $post_meta = get_post_meta($post_id);
                foreach ($post_meta as $key => $values) {
                    foreach ($values as $value) {
                        if (is_email($value)) {
                            $email = sanitize_email($value);
                            break 2;
                        }
                    }
                }
            }
            
            if ($email) {
                error_log('Capitula: Email found in post submission: ' . $email);
                do_action('capitula_form_submitted', $email, 'POST_' . $post->post_type);
            }
        }
    }
    
    /**
     * Strategy 5: Detect Post Status Changes (form entries)
     */
    public function detect_post_status_form_submission($new_status, $old_status, $post) {
        if ($new_status !== 'publish' || $old_status === 'publish') {
            return;
        }
        
        // Use the same logic as detect_post_form_submission
        $this->detect_post_form_submission($post->ID, $post);
    }
    
    /**
     * Handle Normalized Form Submission
     * Central handler for all detected form submissions
     */
    public function handle_normalized_form_submission($email, $source) {
        // Skip WooCommerce requests (WooCommerce sends CPA, not CPL)
        if ($this->is_woocommerce_request()) {
            return;
        }
        
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token || !$email) {
            return;
        }
        
        error_log('Capitula: Normalized form submission - Email: ' . $email . ', Source: ' . $source);
        $this->send_cpl_tracking($email, $token, $source);
    }
    
    /**
     * Extract email from text content
     */
    private function extract_email_from_text($text) {
        $email_pattern = '/[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}/';
        preg_match($email_pattern, $text, $matches);
        
        if (!empty($matches)) {
            return sanitize_email($matches[0]);
        }
        
        return null;
    }
    
    /**
     * Simple CPL Tracking Sender
     */
    private function send_simple_cpl_tracking($email, $token) {
        // Parse token
        if (strpos($token, ':') !== false) {
            $parts = explode(':', $token, 2);
            $ad_position = $parts[0];
            $campaign_token = $parts[1];
        } else {
            $ad_position = substr($token, 0, 1);
            $campaign_token = substr($token, 1);
        }
        
        // Get user IP and geolocation
        $user_ip = $this->get_user_ip();
        $geo_response = wp_remote_get('https://api.ipgeolocation.io/ipgeo?apiKey=d143802d9a6142b493df7085c11d16f6&ip=' . $user_ip, ['timeout' => 10]);
        
        if (is_wp_error($geo_response)) {
            return ['success' => false, 'error' => 'Geolocation failed'];
        }
        
        $geo_data = json_decode(wp_remote_retrieve_body($geo_response), true);
        
        // Prepare CPL data
        $cpl_data = [
            'compaign_token' => $campaign_token,
            'user_email' => $email,
            'compaign_ip' => $geo_data['ip'] ?? $user_ip,
            'ad_position' => $ad_position,
            'device_type' => $this->get_device_type(),
            'compaign_country' => $geo_data['country_name'] ?? ''
        ];
        
        // Send to Capitula API
        $api_response = wp_remote_post('https://api.capitula.co.uk/external-api-cpl', [
            'headers' => ['Content-Type' => 'application/json'],
            'body' => json_encode($cpl_data),
            'timeout' => 30
        ]);
        
        if (is_wp_error($api_response)) {
            error_log('Capitula CPL Error: ' . $api_response->get_error_message());
            return ['success' => false, 'error' => $api_response->get_error_message()];
        }
        
        $response_code = wp_remote_retrieve_response_code($api_response);
        $response_body = wp_remote_retrieve_body($api_response);
        
        error_log('Capitula: CPL sent successfully for ' . $email . ' (HTTP ' . $response_code . ')');
        
        return [
            'success' => true,
            'email' => $email,
            'response_code' => $response_code,
            'message' => 'CPL tracking sent'
        ];
    }
    
    
    /**
     * Universal POST Monitoring - Detects form submissions via POST data
     */
    public function universal_post_monitoring() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            return;
        }
        
        // Skip WooCommerce checkout and admin pages to avoid interfering with CPA
        if (is_admin() || 
            (function_exists('is_checkout') && is_checkout()) ||
            (function_exists('is_wc_endpoint_url') && is_wc_endpoint_url()) ||
            strpos($_SERVER['REQUEST_URI'], '/checkout/') !== false ||
            strpos($_SERVER['REQUEST_URI'], '/wc-api/') !== false ||
            isset($_POST['woocommerce_checkout_place_order']) ||
            isset($_POST['woocommerce-process-checkout-nonce'])) {
            return;
        }
        
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) {
            return;
        }
        
        // TRULY UNIVERSAL: Check ALL $_POST values for valid email addresses
        // This works with ANY form plugin without hardcoding field names
        foreach ($_POST as $field_name => $field_value) {
            // Skip known WooCommerce fields to avoid CPA interference
            if (strpos($field_name, 'woocommerce') !== false || 
                strpos($field_name, 'billing_') !== false ||
                strpos($field_name, 'shipping_') !== false) {
                continue;
            }
            
            // Check if this field contains a valid email
            if (is_string($field_value) && is_email($field_value)) {
                $email = sanitize_email($field_value);
                $this->send_pure_php_cpl_tracking($email, $token);
                error_log('Capitula Universal: Email captured from POST[' . $field_name . '] - ' . $email);
                return; // Exit after first valid email found
            }
        }
        
        // Handle nested arrays (like WPForms)
        foreach ($_POST as $key => $value) {
            if (is_array($value)) {
                foreach ($value as $sub_key => $sub_value) {
                    // Handle deeper nesting (WPForms fields array)
                    if (is_array($sub_value)) {
                        foreach ($sub_value as $deep_key => $deep_value) {
                            if (is_email($deep_value)) {
                                $email = sanitize_email($deep_value);
                                $this->send_pure_php_cpl_tracking($email, $token);
                                error_log('Capitula Universal: Email captured from deep nested POST[' . $key . '][' . $sub_key . '][' . $deep_key . '] - ' . $email);
                                return; // Exit after first capture
                            }
                        }
                    } elseif (is_email($sub_value)) {
                        $email = sanitize_email($sub_value);
                        $this->send_pure_php_cpl_tracking($email, $token);
                        error_log('Capitula Universal: Email captured from nested POST[' . $key . '][' . $sub_key . '] - ' . $email);
                        return; // Exit after first capture
                    }
                }
            }
        }
    }
    
    /**
     * Universal AJAX Monitor - Captures ALL AJAX form submissions
     */
    public function universal_ajax_monitor() {
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) {
            return;
        }
        
        ?>
        <script type="text/javascript">
        document.addEventListener('DOMContentLoaded', function() {
            console.log('Capitula: Universal AJAX monitor initialized');
            
            // Override XMLHttpRequest to catch ALL AJAX submissions
            var originalSend = XMLHttpRequest.prototype.send;
            XMLHttpRequest.prototype.send = function(data) {
                // Check if this is a form submission with email data
                if (data && typeof data === 'string') {
                    var emailRegex = /[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}/g;
                    var emailMatches = data.match(emailRegex);
                    
                    if (emailMatches && emailMatches.length > 0) {
                        // Skip WooCommerce related submissions
                        if (data.indexOf('woocommerce') === -1 && 
                            data.indexOf('billing_') === -1 && 
                            data.indexOf('checkout') === -1) {
                            
                            console.log('Capitula: Email detected in AJAX submission:', emailMatches[0]);
                            
                            // Send CPL tracking
                            fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                                method: 'POST',
                                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                                body: 'action=capitula_universal_cpl&email=' + encodeURIComponent(emailMatches[0]) + '&token=<?php echo esc_js($token); ?>'
                            }).then(function(response) {
                                console.log('Capitula: AJAX CPL sent for', emailMatches[0]);
                            }).catch(function(error) {
                                console.log('Capitula: AJAX CPL error', error);
                            });
                        }
                    }
                }
                
                // Call original send method
                return originalSend.apply(this, arguments);
            };
            
            // Monitor ALL form submissions (including traditional form posts)
            document.addEventListener('submit', function(e) {
                var form = e.target;
                if (form.tagName !== 'FORM') return;
                
                console.log('Capitula: Form submission detected');
                
                // Skip WooCommerce forms
                if (form.classList.contains('woocommerce-checkout') ||
                    form.action.indexOf('checkout') !== -1 ||
                    form.action.indexOf('woocommerce') !== -1) {
                    return;
                }
                
                // Find ANY field with email content
                var formData = new FormData(form);
                for (var pair of formData.entries()) {
                    var value = pair[1];
                    if (typeof value === 'string' && /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value)) {
                        console.log('Capitula: Email found in form submission:', value);
                        
                        // Send CPL tracking
                        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                            method: 'POST',
                            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                            body: 'action=capitula_universal_cpl&email=' + encodeURIComponent(value) + '&token=<?php echo esc_js($token); ?>'
                        }).then(function(response) {
                            console.log('Capitula: Form CPL sent for', value);
                        }).catch(function(error) {
                            console.log('Capitula: Form CPL error', error);
                        });
                        
                        break; // Only capture first email found
                    }
                }
            });
            
            // Also override fetch API
            var originalFetch = window.fetch;
            window.fetch = function() {
                var args = arguments;
                
                // Check if this is a POST request with form data
                if (args.length > 1 && args[1] && args[1].method === 'POST' && args[1].body) {
                    var bodyString = '';
                    
                    if (typeof args[1].body === 'string') {
                        bodyString = args[1].body;
                    } else if (args[1].body instanceof FormData) {
                        // Convert FormData to string for checking
                        var entries = Array.from(args[1].body.entries());
                        bodyString = entries.map(function(pair) {
                            return pair[0] + '=' + pair[1];
                        }).join('&');
                    }
                    
                    if (bodyString) {
                        var emailRegex = /[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}/g;
                        var emailMatches = bodyString.match(emailRegex);
                        
                        if (emailMatches && emailMatches.length > 0) {
                            // Skip WooCommerce related submissions
                            if (bodyString.indexOf('woocommerce') === -1 && 
                                bodyString.indexOf('billing_') === -1 && 
                                bodyString.indexOf('checkout') === -1) {
                                
                                console.log('Capitula: Email detected in Fetch submission:', emailMatches[0]);
                                
                                // Send CPL tracking
                                originalFetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                                    method: 'POST',
                                    headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                                    body: 'action=capitula_universal_cpl&email=' + encodeURIComponent(emailMatches[0]) + '&token=<?php echo esc_js($token); ?>'
                                }).then(function(response) {
                                    console.log('Capitula: Fetch CPL sent for', emailMatches[0]);
                                }).catch(function(error) {
                                    console.log('Capitula: Fetch CPL error', error);
                                });
                            }
                        }
                    }
                }
                
                // Call original fetch method
                return originalFetch.apply(this, arguments);
            };
        });
        </script>
        <?php
    }
    
    /**
     * Universal JavaScript Form Detector - Backup method
     */
    public function universal_js_form_detector() {
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) {
            return;
        }
        
        ?>
        <script type="text/javascript">
        document.addEventListener('DOMContentLoaded', function() {
            // Universal form submission detector
            document.addEventListener('submit', function(e) {
                var form = e.target;
                if (form.tagName !== 'FORM') return;
                
                // Skip WooCommerce checkout forms to avoid CPA interference
                if (form.classList.contains('woocommerce-checkout') ||
                    form.classList.contains('checkout') ||
                    form.classList.contains('cart') ||
                    form.action.indexOf('/checkout/') !== -1 ||
                    form.action.indexOf('wc-api') !== -1 ||
                    form.querySelector('input[name="woocommerce_checkout_place_order"]') ||
                    form.querySelector('input[name*="woocommerce"]') ||
                    document.body.classList.contains('woocommerce-checkout') ||
                    document.body.classList.contains('woocommerce-order-received')) {
                    console.log('Capitula: Skipping WooCommerce form to preserve CPA tracking');
                    return;
                }
                
                // Find ALL input fields and check for valid email values
                var allInputs = form.querySelectorAll('input, textarea, select');
                var emailInputs = [];
                
                // Check each input for email content
                allInputs.forEach(function(input) {
                    if (input.type === 'email' || 
                        input.name.toLowerCase().includes('email') ||
                        input.id.toLowerCase().includes('email') ||
                        input.className.toLowerCase().includes('email') ||
                        (input.value && input.value.includes('@') && /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(input.value))) {
                        emailInputs.push(input);
                    }
                });
                
                emailInputs.forEach(function(input) {
                    if (input.value && input.value.includes('@')) {
                        // Send to our universal handler
                        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                            method: 'POST',
                            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                            body: 'action=capitula_universal_cpl&email=' + encodeURIComponent(input.value) + '&token=<?php echo esc_js($token); ?>'
                        }).then(function(response) {
                            console.log('Capitula Universal: CPL sent for', input.value);
                        }).catch(function(error) {
                            console.log('Capitula Universal: CPL error', error);
                        });
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * WPForms Pure PHP CPL Tracking
     */
    public function wpforms_pure_php_cpl_tracking($fields, $entry, $form_data, $entry_id) {
        foreach ($fields as $field) {
            if (isset($field['type']) && $field['type'] === 'email' && !empty($field['value'])) {
                $email = sanitize_email($field['value']);
                if ($email) {
                    $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
                    if ($token) {
                        $this->send_pure_php_cpl_tracking($email, $token);
                        error_log('Capitula WPForms: CPL sent for ' . $email);
                    }
                    break;
                }
            }
        }
    }
    
    /**
     * WPForms Universal CPL Tracking (backup method for Universal tracking)
     */
    public function wpforms_universal_cpl_tracking($fields, $entry, $form_data, $entry_id) {
        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
        if (!$token) {
            return;
        }
        
        // Find email field in any WPForms submission
        foreach ($fields as $field) {
            if (isset($field['type']) && $field['type'] === 'email' && !empty($field['value'])) {
                $email = sanitize_email($field['value']);
                if ($email) {
                    $this->send_pure_php_cpl_tracking($email, $token);
                    error_log('Capitula WPForms Universal: CPL sent for ' . $email);
                    return; // Exit after first email found
                }
            }
        }
        
        // Fallback: check raw entry data for any email values
        if (is_array($entry)) {
            foreach ($entry as $field_id => $field_value) {
                if (is_email($field_value)) {
                    $email = sanitize_email($field_value);
                    $this->send_pure_php_cpl_tracking($email, $token);
                    error_log('Capitula WPForms Universal Fallback: CPL sent for ' . $email);
                    return;
                }
            }
        }
    }
    
    /**
     * FluentForms Pure PHP CPL Tracking
     */
    public function fluentforms_pure_php_cpl_tracking($entry_id, $form_data, $form) {
        if (isset($form_data) && is_array($form_data)) {
            foreach ($form_data as $field => $value) {
                if (is_email($value)) {
                    $email = sanitize_email($value);
                    $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
                    if ($token) {
                        $this->send_pure_php_cpl_tracking($email, $token);
                        error_log('Capitula FluentForms: CPL sent for ' . $email);
                    }
                    break;
                }
            }
        }
    }
    
    /**
     * JotForm Webhook Listener
     */
    public function jotform_webhook_listener() {
        if (isset($_GET['capitula_jotform_webhook'])) {
            $json_input = file_get_contents('php://input');
            $webhook_data = json_decode($json_input, true);
            
            if (isset($webhook_data['rawRequest'])) {
                parse_str($webhook_data['rawRequest'], $form_data);
                
                foreach ($form_data as $key => $value) {
                    if (is_email($value)) {
                        $email = sanitize_email($value);
                        $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
                        if ($token) {
                            $this->send_pure_php_cpl_tracking($email, $token);
                            error_log('Capitula JotForm: CPL sent for ' . $email);
                        }
                        break;
                    }
                }
            }
            
            wp_die('OK', 200);
        }
    }
    
    /**
     * Send Pure PHP CPL Tracking (No JavaScript)
     */
    private function send_pure_php_cpl_tracking($email, $token) {
        // Parse token - handle colon-separated format
        if (strpos($token, ':') !== false) {
            // New format: "position:token"
            $parts = explode(':', $token, 2);
            $ad_position = $parts[0];
            $campaign_token = $parts[1];
        } else {
            // Legacy format: first char = ad_position, rest = campaign_token
            $ad_position = substr($token, 0, 1);
            $campaign_token = substr($token, 1);
        }
        
        // Get user IP
        $user_ip = $this->get_user_ip();
        
        // Get geolocation
        $geo_response = wp_remote_get('https://api.ipgeolocation.io/ipgeo?apiKey=d143802d9a6142b493df7085c11d16f6&ip=' . $user_ip, [
            'timeout' => 10
        ]);
        
        if (!is_wp_error($geo_response)) {
            $geo_data = json_decode(wp_remote_retrieve_body($geo_response), true);
            
            // Prepare CPL data
            $cpl_data = [
                'compaign_token' => $campaign_token,
                'user_email' => $email,
                'compaign_ip' => $geo_data['ip'] ?? $user_ip,
                'ad_position' => $ad_position,
                'device_type' => $this->get_device_type(),
                'compaign_country' => $geo_data['country_name'] ?? ''
            ];
            
            // Send to Capitula API (non-blocking to avoid slowing down form)
            wp_remote_post('https://api.capitula.co.uk/external-api-cpl', [
                'headers' => ['Content-Type' => 'application/json'],
                'body' => json_encode($cpl_data),
                'timeout' => 30,
                'blocking' => false // Non-blocking so it doesn't slow down the form
            ]);
            
            // Log for debugging
            error_log('Capitula Pure PHP CPL sent for: ' . $email . ' with token: ' . $campaign_token);
        } else {
            error_log('Capitula Pure PHP CPL failed - geolocation error: ' . $geo_response->get_error_message());
        }
    }
    
    /**
     * Output stored form emails for tracking
     * This runs in wp_footer to avoid interfering with AJAX responses
     */
    public function output_stored_form_emails() {
        $session_id = session_id();
        
        // Check for Contact Form 7 email after page refresh
        $cf7_display_email = get_transient('capitula_cf7_display_email_' . $session_id);
        if ($cf7_display_email) {
            // Show thank you message and email tracking
            ?>
            <div id="capitula-cf7-success" style="background: #d4edda; border: 1px solid #c3e6cb; color: #155724; padding: 15px; margin: 20px 0; border-radius: 4px;">
                <strong>Thank you for your message. It has been sent.</strong><br>
                Email submitted: <?php echo esc_html($cf7_display_email); ?>
            </div>
            
            <div style="display:none;">
                <capitula_email><?php echo esc_html($cf7_display_email); ?></capitula_email>
            </div>
            
            <script type="text/javascript">
            document.addEventListener('DOMContentLoaded', function() {
                // Scroll to the success message
                var successDiv = document.getElementById('capitula-cf7-success');
                if (successDiv) {
                    successDiv.scrollIntoView({ behavior: 'smooth' });
                }
            });
            </script>
            <?php
            delete_transient('capitula_cf7_display_email_' . $session_id);
        }
        
        // Check for Gravity Forms email
        $gf_email = get_transient('capitula_gf_email_' . $session_id);
        if ($gf_email) {
            echo '<div style="display:none;">';
            echo '<capitula_email>' . esc_html($gf_email) . '</capitula_email>';
            echo '</div>';
            delete_transient('capitula_gf_email_' . $session_id);
        }
        
        // Check for Ninja Forms email
        $nf_email = get_transient('capitula_nf_email_' . $session_id);
        if ($nf_email) {
            echo '<div style="display:none;">';
            echo '<capitula_email>' . esc_html($nf_email) . '</capitula_email>';
            echo '</div>';
            delete_transient('capitula_nf_email_' . $session_id);
        }
    }
    
    /**
     * Gravity Forms tracking
     */
    public function gravity_forms_tracking($entry, $form) {
        foreach ($form['fields'] as $field) {
            if ($field->type === 'email' && !empty($entry[$field->id])) {
                $email = sanitize_email($entry[$field->id]);
                if ($email) {
                    // Store email in transient to avoid interfering with form responses
                    set_transient('capitula_gf_email_' . session_id(), $email, 300); // 5 minutes
                    break;
                }
            }
        }
    }
    
    /**
     * Ninja Forms tracking
     */
    public function ninja_forms_tracking($form_data) {
        if (isset($form_data['fields'])) {
            foreach ($form_data['fields'] as $field) {
                if ($field['type'] === 'email' && !empty($field['value'])) {
                    $email = sanitize_email($field['value']);
                    if ($email) {
                        // Store email in transient to avoid interfering with form responses
                        set_transient('capitula_nf_email_' . session_id(), $email, 300); // 5 minutes
                        break;
                    }
                }
            }
        }
    }
    
    /**
     * Add custom CSS to admin head
     */
    public function admin_custom_css() {
        $screen = get_current_screen();
        // Only load on Capitula admin pages
        if (strpos($screen->id, 'capitula') !== false) {
            ?>
            <style>
            /* Custom Capitula Admin Styling */
            .capitula-brand-color {
                color: #25caf1 !important;
            }
            .capitula-border {
                border-color: #25caf1 !important;
            }
            .capitula-bg {
                background-color: #f0fcff !important;
            }
            </style>
            <?php
        }
        
        // Add custom menu icon styling
        ?>
        <style>
        /* Custom Capitula Menu Icon */
        #adminmenu .menu-icon-capitula-campaign-config .wp-menu-image {
            background-image: url('<?php echo CAPITULA_PLUGIN_URL; ?>logo-black.png') !important;
            background-size: 20px 20px !important;
            background-repeat: no-repeat !important;
            background-position: center !important;
        }
        #adminmenu .menu-icon-capitula-campaign-config .wp-menu-image:before {
            content: '' !important;
        }
        #adminmenu .menu-icon-capitula-campaign-config:hover .wp-menu-image,
        #adminmenu .menu-icon-capitula-campaign-config.wp-has-current-submenu .wp-menu-image {
            filter: brightness(0) invert(1) !important;
        }
        </style>
        <?php
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            'Capitula Campaign Configuration',
            'Capitula',
            'manage_options',
            'capitula-campaign-config',
            array($this, 'admin_page'),
            'none', // Custom icon via CSS
            30 // Position after Comments
        );
        
        // Add submenu items
        add_submenu_page(
            'capitula-campaign-config',
            'Settings',
            'Settings', 
            'manage_options',
            'capitula-campaign-config',
            array($this, 'admin_page')
        );
        
        add_submenu_page(
            'capitula-campaign-config',
            'Status & Debug',
            'Status & Debug',
            'manage_options',
            'capitula-status',
            array($this, 'status_page')
        );
    }
    
    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting('capitula_settings', 'capitula_enable_fallback');
        register_setting('capitula_settings', 'capitula_enable_woocommerce');
        register_setting('capitula_settings', 'capitula_enable_contact_forms');
        register_setting('capitula_settings', 'capitula_enable_registration');
        register_setting('capitula_settings', 'capitula_cpl_tracking_method');
        register_setting('capitula_settings', 'capitula_configured_form_plugins');
        register_setting('capitula_settings', 'capitula_universal_tracking_enabled');
    }
    
    /**
     * Handle quick method switching via URL parameters
     */
    private function handle_quick_method_switch() {
        // Handle testing simple_ajax if CPA issues persist
        if (isset($_GET['test_simple_ajax']) && current_user_can('manage_options')) {
            update_option('capitula_cpl_tracking_method', 'simple_ajax');
            // Keep universal tracking enabled
            
            // Simple redirect without admin notices to avoid conflicts
            wp_redirect(admin_url('admin.php?page=capitula-campaign-config&switched=simple_ajax'));
            exit;
        }
        
        if (isset($_GET['switch_method']) && current_user_can('manage_options')) {
            $method = sanitize_text_field($_GET['switch_method']);
            $valid_methods = ['simple_ajax', 'pure_php', 'webhook_api', 'database_queue'];
            
            if (in_array($method, $valid_methods)) {
                update_option('capitula_cpl_tracking_method', $method);
                
                // Simple redirect without admin notices to avoid conflicts
                wp_redirect(admin_url('admin.php?page=capitula-campaign-config&switched=' . $method));
                exit;
            }
        }
    }
    
    /**
     * Main admin page display - Settings
     */
    public function admin_page() {
        $this->handle_quick_method_switch();
        ?>
        <div class="wrap">
            <!-- Header Section -->
            <div style="background: #fff; padding: 20px; margin: 0 0 20px 0; border: 1px solid #ccd0d4; box-shadow: 0 1px 1px rgba(0,0,0,.04);">
                <div style="display: flex; align-items: center; margin-bottom: 0;">
                    <img src="<?php echo CAPITULA_PLUGIN_URL; ?>logo-black.png" alt="Capitula" style="height: 40px; margin-right: 15px;" />
                    <h1 style="margin: 0; color: #23282d; font-size: 28px;">Campaign Configuration</h1>
                </div>
            </div>
            
            <!-- Notifications Section -->
            <?php if (isset($_GET['settings-updated'])): ?>
            <div style="margin: 0 0 20px 0;">
                <div class="notice notice-success is-dismissible" style="margin: 0 0 10px 0;">
                    <p><span class="dashicons dashicons-saved"></span> Settings saved successfully!</p>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- Main Content Section -->
            <div class="capitula-admin-container" style="display: flex; gap: 20px; margin: 0;">
                <!-- Main Settings Panel -->
                <div class="capitula-main-panel" style="flex: 2; background: #fff; padding: 20px; border: 1px solid #ccd0d4; box-shadow: 0 1px 1px rgba(0,0,0,.04);">
                    <h2 style="margin-top: 0; color: #23282d; border-bottom: 2px solid #25caf1; padding-bottom: 10px;"><span class="dashicons dashicons-admin-settings"></span> Configuration</h2>
                    
                    <form method="post" action="options.php">
                        <?php
                        settings_fields('capitula_settings');
                        do_settings_sections('capitula_settings');
                        ?>
                        
                        <table class="form-table" style="margin-top: 20px;">
                            <tr>
                                <th scope="row"><span class="dashicons dashicons-shield-alt"></span> Cross-Domain Fallback</th>
                                <td>
                                    <fieldset>
                                        <label style="display: flex; align-items: center; gap: 8px;">
                                            <input type="checkbox" name="capitula_enable_fallback" value="yes" <?php checked(get_option('capitula_enable_fallback', 'no'), 'yes'); ?> />
                                            <strong>Enable fallback loading</strong>
                                        </label>
                                        <p class="description" style="margin-top: 8px; margin-left: 24px; color: #646970;">Use modern script loading (no document.write) to prevent browser warnings and cross-domain issues</p>
                                    </fieldset>
                                </td>
                            </tr>
                        </table>
                        
                        <h3 style="margin-top: 30px; color: #23282d; border-bottom: 1px solid #ddd; padding-bottom: 5px;"><span class="dashicons dashicons-admin-tools"></span> Tracking Features</h3>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row"><span class="dashicons dashicons-cart"></span> WooCommerce</th>
                                <td>
                                    <fieldset>
                                        <label style="display: flex; align-items: center; gap: 8px;">
                                            <input type="checkbox" name="capitula_enable_woocommerce" value="yes" <?php checked(get_option('capitula_enable_woocommerce', 'yes'), 'yes'); ?> />
                                            <strong>Order conversion tracking</strong>
                                        </label>
                                        <p class="description" style="margin-top: 8px; margin-left: 24px; color: #646970;">Track purchases with &lt;capitula_pay&gt; and &lt;capitula_trans&gt; tags</p>
                                    </fieldset>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row"><span class="dashicons dashicons-email-alt"></span> Contact Forms</th>
                                <td>
                                    <fieldset>
                                        <label style="display: flex; align-items: center; gap: 8px;">
                                            <input type="checkbox" name="capitula_enable_contact_forms" value="yes" <?php checked(get_option('capitula_enable_contact_forms', 'yes'), 'yes'); ?> />
                                            <strong>Form submission tracking</strong>
                                        </label>
                                        <p class="description" style="margin-top: 8px; margin-left: 24px; color: #646970;">Supports Contact Form 7, Gravity Forms, and Ninja Forms</p>
                                    </fieldset>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row"><span class="dashicons dashicons-admin-users"></span> User Registration</th>
                                <td>
                                    <fieldset>
                                        <label style="display: flex; align-items: center; gap: 8px;">
                                            <input type="checkbox" name="capitula_enable_registration" value="yes" <?php checked(get_option('capitula_enable_registration', 'yes'), 'yes'); ?> />
                                            <strong>Registration conversion tracking</strong>
                                        </label>
                                        <p class="description" style="margin-top: 8px; margin-left: 24px; color: #646970;">Track new user registrations with &lt;capitula_email&gt; tags</p>
                                    </fieldset>
                                </td>
                            </tr>
                        </table>
                        
                        <h3 style="margin-top: 30px; color: #23282d; border-bottom: 1px solid #ddd; padding-bottom: 5px;"><span class="dashicons dashicons-performance"></span> CPL Tracking Method</h3>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row"><span class="dashicons dashicons-yes-alt"></span> Universal Tracking</th>
                                <td>
                                    <fieldset>
                                        <?php $universal_enabled = get_option('capitula_universal_tracking_enabled', 'yes'); ?>
                                        <label style="display: flex; align-items: center; gap: 8px;">
                                            <input type="checkbox" name="capitula_universal_tracking_enabled" value="yes" <?php checked($universal_enabled, 'yes'); ?> />
                                            <strong>Enable Universal Form Detection</strong>
                                        </label>
                                        <p class="description" style="margin-top: 8px; margin-left: 24px; color: #646970;">✅ Works with <strong>ANY form plugin</strong> (WPForms, Contact Form 7, Gravity Forms, etc.)<br>✅ Automatically detects email submissions without manual configuration<br>🔧 <strong>Required for WPForms CPL tracking</strong></p>
                                    </fieldset>
                                </td>
                            </tr>
                        </table>
                        
                        <h3 style="margin-top: 30px; color: #23282d; border-bottom: 1px solid #ddd; padding-bottom: 5px;"><span class="dashicons dashicons-admin-tools"></span> Advanced: CPL Tracking Method</h3>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row"><span class="dashicons dashicons-admin-tools"></span> Tracking Method</th>
                                <td>
                                    <fieldset>
                                        <?php $current_method = get_option('capitula_cpl_tracking_method', 'simple_ajax'); ?>
                                        
                                        <p style="margin-bottom: 15px; color: #646970; font-weight: 500;">Choose how CPL (Cost Per Lead) tracking should be handled:</p>
                                        
                                        <div style="margin-bottom: 12px;">
                                            <label style="display: flex; align-items: flex-start; gap: 8px; padding: 12px; border: 2px solid <?php echo ($current_method === 'simple_ajax') ? '#25caf1' : '#ddd'; ?>; border-radius: 6px; background: <?php echo ($current_method === 'simple_ajax') ? '#f0fcff' : '#fff'; ?>;">
                                                <input type="radio" name="capitula_cpl_tracking_method" value="simple_ajax" <?php checked($current_method, 'simple_ajax'); ?> style="margin-top: 2px;" />
                                                <div>
                                                    <strong style="color: #25caf1;">🚀 Simple AJAX (Recommended)</strong><br>
                                                    <span style="color: #646970; font-size: 13px;">Lightweight JavaScript + server-side processing. Good balance of reliability and performance.</span>
                                                </div>
                                            </label>
                                        </div>
                                        
                                        <div style="margin-bottom: 12px;">
                                            <label style="display: flex; align-items: flex-start; gap: 8px; padding: 12px; border: 2px solid <?php echo ($current_method === 'pure_php') ? '#25caf1' : '#ddd'; ?>; border-radius: 6px; background: <?php echo ($current_method === 'pure_php') ? '#f0fcff' : '#fff'; ?>;">
                                                <input type="radio" name="capitula_cpl_tracking_method" value="pure_php" <?php checked($current_method, 'pure_php'); ?> style="margin-top: 2px;" />
                                                <div>
                                                    <strong style="color: #00a32a;">🛡️ Pure PHP (Most Reliable)</strong><br>
                                                    <span style="color: #646970; font-size: 13px;">100% server-side processing. No JavaScript dependencies, zero network issues.</span>
                                                </div>
                                            </label>
                                        </div>
                                        
                                        
                                        <div style="margin-top: 15px; padding: 12px; background: #fff3cd; border: 1px solid #ffeaa7; border-radius: 4px;">
                                            <strong style="color: #856404;">💡 Switching Methods:</strong><br>
                                            <span style="color: #856404; font-size: 13px;">Changes take effect immediately. If you experience form submission issues, switch to "Pure PHP" method.</span>
                                        </div>
                                        
                                    </fieldset>
                                </td>
                            </tr>
                        </table>
                        
                        <?php submit_button('Save Settings', 'primary', 'submit', true, ['style' => 'margin-top: 20px; font-size: 14px;']); ?>
                    </form>
                </div>
                
                <!-- Quick Status Sidebar -->
                <div class="capitula-sidebar" style="flex: 1; background: #fff; padding: 20px; border: 1px solid #ccd0d4; box-shadow: 0 1px 1px rgba(0,0,0,.04); height: fit-content;">
                    <h3 style="margin-top: 0; color: #23282d;"><span class="dashicons dashicons-dashboard"></span> Quick Status</h3>
                    
                    <div style="margin-bottom: 15px; padding: 10px; background: #f0fcff; border-left: 4px solid #25caf1;">
                        <strong>Plugin Version:</strong><br>
                        <span style="font-family: monospace;"><?php echo CAPITULA_PLUGIN_VERSION; ?></span>
                    </div>
                    
                    <div style="margin-bottom: 15px; padding: 10px; background: <?php echo class_exists('WooCommerce') ? '#d1eddb' : '#fcf2cd'; ?>; border-left: 4px solid <?php echo class_exists('WooCommerce') ? '#00a32a' : '#dba617'; ?>;">
                        <strong>WooCommerce:</strong><br>
                        <?php if (class_exists('WooCommerce')): ?>
                            <span style="color: #00a32a;">✅ Active (v<?php echo WC()->version; ?>)</span>
                        <?php else: ?>
                            <span style="color: #dba617;">⚠️ Not Installed</span>
                        <?php endif; ?>
                    </div>
                    
                    <div style="margin-bottom: 15px; padding: 10px; background: <?php echo defined('WPCF7_VERSION') ? '#d1eddb' : '#f6f7f7'; ?>; border-left: 4px solid <?php echo defined('WPCF7_VERSION') ? '#00a32a' : '#8c8f94'; ?>;">
                        <strong>Contact Form 7:</strong><br>
                        <?php if (defined('WPCF7_VERSION')): ?>
                            <span style="color: #00a32a;">✅ Active (v<?php echo WPCF7_VERSION; ?>)</span>
                        <?php else: ?>
                            <span style="color: #8c8f94;">➖ Not Installed</span>
                        <?php endif; ?>
                    </div>
                    
                    <?php $current_method = get_option('capitula_cpl_tracking_method', 'simple_ajax'); ?>
                    <div style="margin-bottom: 15px; padding: 10px; background: #f0fcff; border-left: 4px solid #25caf1;">
                        <strong>CPL Tracking:</strong><br>
                        <?php 
                        switch ($current_method) {
                            case 'simple_ajax':
                                echo '<span style="color: #25caf1;">🚀 Simple AJAX</span>';
                                break;
                            case 'pure_php':
                                echo '<span style="color: #00a32a;">🛡️ Pure PHP</span>';
                                break;
                            default:
                                echo '<span style="color: #00a32a;">🛡️ Pure PHP</span>';
                        }
                        ?>
                    </div>
                    
                    <hr style="margin: 20px 0;">
                    
                    <p style="text-align: center; margin-bottom: 15px;">
                        <a href="<?php echo admin_url('admin.php?page=capitula-status'); ?>" class="button button-secondary" style="width: 100%;">
                            <span class="dashicons dashicons-admin-tools"></span> View Full Status
                        </a>
                    </p>
                    
                    <!-- Security Notice -->
                    <div style="margin-top: 20px; padding: 15px; background: #e8f5e9; border: 1px solid #4caf50; border-radius: 4px;">
                        <div style="display: flex; align-items: flex-start; gap: 10px;">
                            <span class="dashicons dashicons-lock" style="color: #4caf50; font-size: 20px; margin-top: 2px;"></span>
                            <div>
                                <strong style="color: #2e7d32; font-size: 14px; display: block; margin-bottom: 8px;">Security Notice: Payment Data</strong>
                                <p style="margin: 0; color: #1b5e20; font-size: 13px; line-height: 1.5;">
                                    The Capitula Campaign Configuration plugin is strictly for analytics and tracking (like CDN Integration and Universal Form Tracking). We <strong>NEVER</strong> capture or store customer credit card or payment details. All payment processing is the sole responsibility of your designated, secure payment gateway.
                                </p>
                            </div>
                        </div>
                    </div>
                    
                </div>
            </div>
            
        </div>
        
        <style>
        .capitula-admin-container table th {
            font-weight: 600;
            color: #23282d;
        }
        .capitula-admin-container .dashicons {
            margin-right: 5px;
            color: #25caf1;
        }
        .capitula-admin-container fieldset label {
            font-size: 14px;
        }
        @media (max-width: 1200px) {
            .capitula-admin-container {
                flex-direction: column !important;
            }
        }
        </style>
        <?php
    }
    
    /**
     * Status & Debug page
     */
    public function status_page() {
        ?>
        <div class="wrap">
            <!-- Header Section -->
            <div style="background: #fff; padding: 20px; margin: 0 0 20px 0; border: 1px solid #ccd0d4; box-shadow: 0 1px 1px rgba(0,0,0,.04);">
                <div style="display: flex; align-items: center; margin-bottom: 0;">
                    <img src="<?php echo CAPITULA_PLUGIN_URL; ?>logo-black.png" alt="Capitula" style="height: 40px; margin-right: 15px;" />
                    <h1 style="margin: 0; color: #23282d; font-size: 28px;">Status & Debug</h1>
                </div>
            </div>
            
            <div style="display: flex; gap: 20px; margin: 0;">
                <!-- System Information -->
                <div style="flex: 1; background: #fff; padding: 20px; border: 1px solid #ccd0d4; box-shadow: 0 1px 1px rgba(0,0,0,.04);">
                    <h2 style="margin-top: 0; color: #23282d; border-bottom: 2px solid #25caf1; padding-bottom: 10px;"><span class="dashicons dashicons-admin-settings"></span> System Information</h2>
                    
                    <table class="widefat striped" style="margin-top: 15px;">
                        <thead>
                            <tr>
                                <th style="width: 30%;">Component</th>
                                <th>Status</th>
                                <th>Version/Details</th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td><strong>WordPress</strong></td>
                                <td><span style="color: #00a32a;">✅ Active</span></td>
                                <td><?php echo get_bloginfo('version'); ?></td>
                            </tr>
                            <tr>
                                <td><strong>Capitula Plugin</strong></td>
                                <td><span style="color: #00a32a;">✅ Active</span></td>
                                <td><?php echo CAPITULA_PLUGIN_VERSION; ?></td>
                            </tr>
                            <tr>
                                <td><strong>WooCommerce</strong></td>
                                <td>
                                    <?php if (class_exists('WooCommerce')): ?>
                                        <span style="color: #00a32a;">✅ Active</span>
                                    <?php else: ?>
                                        <span style="color: #dba617;">⚠️ Not Installed</span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo class_exists('WooCommerce') ? WC()->version : 'Not Available'; ?></td>
                            </tr>
                            <?php 
                            // Detect ANY form-related plugins dynamically
                            $detected_form_plugins = [];
                            
                            // Get all active plugins
                            $active_plugins = get_option('active_plugins', []);
                            $all_plugins = get_plugins();
                            
                            // Form-related keywords to identify form plugins
                            $form_keywords = ['form', 'contact', 'gravity', 'ninja', 'fluent', 'elementor', 'wpforms', 'jotform', 'typeform', 'mailchimp', 'convertkit'];
                            
                            foreach ($active_plugins as $plugin_path) {
                                if (isset($all_plugins[$plugin_path])) {
                                    $plugin_data = $all_plugins[$plugin_path];
                                    $plugin_name = $plugin_data['Name'];
                                    $plugin_name_lower = strtolower($plugin_name);
                                    
                                    // Check if plugin name contains form-related keywords
                                    $is_form_plugin = false;
                                    foreach ($form_keywords as $keyword) {
                                        if (strpos($plugin_name_lower, $keyword) !== false) {
                                            $is_form_plugin = true;
                                            break;
                                        }
                                    }
                                    
                                    // Also check plugin description for form-related content
                                    if (!$is_form_plugin && !empty($plugin_data['Description'])) {
                                        $description_lower = strtolower($plugin_data['Description']);
                                        foreach ($form_keywords as $keyword) {
                                            if (strpos($description_lower, $keyword) !== false) {
                                                $is_form_plugin = true;
                                                break;
                                            }
                                        }
                                    }
                                    
                                    if ($is_form_plugin) {
                                        $detected_form_plugins[] = [
                                            'name' => $plugin_name,
                                            'version' => $plugin_data['Version'] ?? 'Unknown'
                                        ];
                                    }
                                }
                            }
                            
                            // If no form plugins detected, show a message
                            if (empty($detected_form_plugins)) {
                            ?>
                            <tr>
                                <td colspan="3" style="text-align: center; color: #8c8f94; font-style: italic;">
                                    No form plugins detected. Universal tracking will work with any form submissions.
                                </td>
                            </tr>
                            <?php 
                            } else {
                                // Show detected form plugins
                                foreach ($detected_form_plugins as $plugin) {
                            ?>
                            <tr>
                                <td><strong><?php echo esc_html($plugin['name']); ?></strong></td>
                                <td><span style="color: #00a32a;">✅ Active</span></td>
                                <td><?php echo esc_html($plugin['version']); ?></td>
                            </tr>
                            <?php 
                                }
                            } 
                            ?>
                        </tbody>
                    </table>
                </div>
                
                <!-- Current Settings -->
                <div style="flex: 1; background: #fff; padding: 20px; border: 1px solid #ccd0d4; box-shadow: 0 1px 1px rgba(0,0,0,.04);">
                    <h2 style="margin-top: 0; color: #23282d; border-bottom: 2px solid #25caf1; padding-bottom: 10px;"><span class="dashicons dashicons-admin-generic"></span> Current Configuration</h2>
                    
                    <table class="widefat striped" style="margin-top: 15px;">
                        <thead>
                            <tr>
                                <th>Setting</th>
                                <th>Value</th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td><strong>CDN Base URL</strong></td>
                                <td><code>https://api.capitula.co.uk</code> <em>(Fixed)</em></td>
                            </tr>
                            <tr>
                                <td><strong>Cross-Domain Fallback</strong></td>
                                <td>
                                    <?php $fallback = get_option('capitula_enable_fallback', 'no'); ?>
                                    <span style="color: <?php echo $fallback === 'yes' ? '#00a32a' : '#8c8f94'; ?>;">
                                        <?php echo $fallback === 'yes' ? '✅ Enabled' : '➖ Disabled'; ?>
                                    </span>
                                </td>
                            </tr>
                            <tr>
                                <td><strong>WooCommerce Tracking</strong></td>
                                <td>
                                    <?php $woo = get_option('capitula_enable_woocommerce', 'yes'); ?>
                                    <span style="color: <?php echo $woo === 'yes' ? '#00a32a' : '#8c8f94'; ?>;">
                                        <?php echo $woo === 'yes' ? '✅ Enabled' : '➖ Disabled'; ?>
                                    </span>
                                </td>
                            </tr>
                            <tr>
                                <td><strong>Contact Form Tracking</strong></td>
                                <td>
                                    <?php $forms = get_option('capitula_enable_contact_forms', 'yes'); ?>
                                    <span style="color: <?php echo $forms === 'yes' ? '#00a32a' : '#8c8f94'; ?>;">
                                        <?php echo $forms === 'yes' ? '✅ Enabled' : '➖ Disabled'; ?>
                                    </span>
                                </td>
                            </tr>
                            <tr>
                                <td><strong>Registration Tracking</strong></td>
                                <td>
                                    <?php $reg = get_option('capitula_enable_registration', 'yes'); ?>
                                    <span style="color: <?php echo $reg === 'yes' ? '#00a32a' : '#8c8f94'; ?>;">
                                        <?php echo $reg === 'yes' ? '✅ Enabled' : '➖ Disabled'; ?>
                                    </span>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
            
        </div>
        <?php
    }
    
    /**
     * Form Setup Wizard Page
     */
    public function form_wizard_page() {
        // Handle form submission
        if (isset($_POST['submit_wizard']) && wp_verify_nonce($_POST['capitula_wizard_nonce'], 'capitula_wizard_action')) {
            $configured_plugins = array();
            $universal_enabled = isset($_POST['enable_universal']) ? 'yes' : 'no';
            
            // Save selected form plugins
            if (isset($_POST['form_plugins']) && is_array($_POST['form_plugins'])) {
                $configured_plugins = array_map('sanitize_text_field', $_POST['form_plugins']);
            }
            
            update_option('capitula_configured_form_plugins', $configured_plugins);
            update_option('capitula_universal_tracking_enabled', $universal_enabled);
            
            echo '<div class="notice notice-success"><p><strong>Form Setup Complete!</strong> Your configuration has been saved.</p></div>';
        }
        
        $configured_plugins = get_option('capitula_configured_form_plugins', array());
        $universal_enabled = get_option('capitula_universal_tracking_enabled', 'yes');
        
        ?>
        <div class="wrap">
            <!-- Header Section -->
            <div style="background: #fff; padding: 20px; margin: 0 0 20px 0; border: 1px solid #ccd0d4; box-shadow: 0 1px 1px rgba(0,0,0,.04);">
                <div style="display: flex; align-items: center; margin-bottom: 0;">
                    <img src="<?php echo CAPITULA_PLUGIN_URL; ?>logo-black.png" alt="Capitula" style="height: 40px; margin-right: 15px;" />
                    <h1 style="margin: 0; color: #23282d; font-size: 28px;">Form Setup Wizard</h1>
                </div>
            </div>
            
            <div style="display: flex; gap: 20px;">
                <!-- Main Wizard Panel -->
                <div style="flex: 2; background: #fff; padding: 30px; border: 1px solid #ccd0d4; box-shadow: 0 1px 1px rgba(0,0,0,.04);">
                    <h2 style="margin-top: 0; color: #23282d; border-bottom: 2px solid #25caf1; padding-bottom: 10px;">
                        <span class="dashicons dashicons-admin-tools"></span> Configure Form Tracking
                    </h2>
                    
                    <p style="font-size: 16px; color: #555; line-height: 1.6;">This wizard helps you set up CPL (Cost Per Lead) tracking for your forms. Choose how you want to track form submissions:</p>
                    
                    <form method="post" action="">
                        <?php wp_nonce_field('capitula_wizard_action', 'capitula_wizard_nonce'); ?>
                        
                        <!-- Universal Tracking Option -->
                        <div style="margin: 25px 0; padding: 20px; background: #e8f5e8; border: 2px solid #4CAF50; border-radius: 8px;">
                            <h3 style="margin-top: 0; color: #2e7d32;"><span class="dashicons dashicons-yes-alt"></span> Universal Tracking (Recommended)</h3>
                            <label style="display: flex; align-items: center; gap: 10px; font-size: 16px;">
                                <input type="checkbox" name="enable_universal" value="yes" <?php checked($universal_enabled, 'yes'); ?> style="transform: scale(1.2);" />
                                <strong>Enable Universal Form Detection</strong>
                            </label>
                            <p style="margin: 15px 0 0 32px; color: #555; line-height: 1.5;">✅ Works with <strong>ANY form plugin</strong> (WPForms, Contact Form 7, Gravity Forms, Ninja Forms, Fluent Forms, JotForm, etc.)<br>
                            ✅ Automatically detects email submissions<br>
                            ✅ No manual configuration needed<br>
                            ✅ Future-proof for new form plugins</p>
                        </div>
                        
                        <div style="margin: 20px 0; padding: 15px; background: #fff3cd; border-left: 4px solid #ffc107; border-radius: 4px;">
                            <p style="margin: 0; color: #856404;"><strong>💡 Pro Tip:</strong> Universal Tracking is recommended for most users. It automatically works with any form plugin without manual setup.</p>
                        </div>
                        
                        <hr style="margin: 30px 0;">
                        
                        <!-- Specific Plugin Configuration -->
                        <h3 style="color: #23282d; margin-bottom: 20px;"><span class="dashicons dashicons-admin-plugins"></span> Plugin-Specific Integration (Optional)</h3>
                        <p style="color: #666; margin-bottom: 20px;">For advanced users: Enable specific plugin hooks for enhanced integration. Universal tracking will still work as a backup.</p>
                        
                        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 15px; margin: 20px 0;">
                            
                            <!-- Contact Form 7 -->
                            <?php if (defined('WPCF7_VERSION')): ?>
                            <div style="padding: 15px; border: 2px solid <?php echo in_array('contact_form_7', $configured_plugins) ? '#25caf1' : '#ddd'; ?>; border-radius: 8px; background: <?php echo in_array('contact_form_7', $configured_plugins) ? '#f0fcff' : '#fff'; ?>;">
                                <label style="display: flex; align-items: center; gap: 10px;">
                                    <input type="checkbox" name="form_plugins[]" value="contact_form_7" <?php checked(in_array('contact_form_7', $configured_plugins)); ?> />
                                    <strong>Contact Form 7</strong>
                                    <span style="color: #00a32a; font-size: 12px;">✅ Active (v<?php echo WPCF7_VERSION; ?>)</span>
                                </label>
                                <p style="margin: 8px 0 0 32px; color: #666; font-size: 13px;">Enhanced hook: wpcf7_mail_sent</p>
                            </div>
                            <?php endif; ?>
                            
                            <!-- WPForms -->
                            <?php if (function_exists('wpforms')): ?>
                            <div style="padding: 15px; border: 2px solid <?php echo in_array('wpforms', $configured_plugins) ? '#25caf1' : '#ddd'; ?>; border-radius: 8px; background: <?php echo in_array('wpforms', $configured_plugins) ? '#f0fcff' : '#fff'; ?>;">
                                <label style="display: flex; align-items: center; gap: 10px;">
                                    <input type="checkbox" name="form_plugins[]" value="wpforms" <?php checked(in_array('wpforms', $configured_plugins)); ?> />
                                    <strong>WPForms</strong>
                                    <span style="color: #00a32a; font-size: 12px;">✅ Active</span>
                                </label>
                                <p style="margin: 8px 0 0 32px; color: #666; font-size: 13px;">Enhanced hook: wpforms_process_complete</p>
                            </div>
                            <?php endif; ?>
                            
                            <!-- Gravity Forms -->
                            <?php if (class_exists('GFForms')): ?>
                            <div style="padding: 15px; border: 2px solid <?php echo in_array('gravity_forms', $configured_plugins) ? '#25caf1' : '#ddd'; ?>; border-radius: 8px; background: <?php echo in_array('gravity_forms', $configured_plugins) ? '#f0fcff' : '#fff'; ?>;">
                                <label style="display: flex; align-items: center; gap: 10px;">
                                    <input type="checkbox" name="form_plugins[]" value="gravity_forms" <?php checked(in_array('gravity_forms', $configured_plugins)); ?> />
                                    <strong>Gravity Forms</strong>
                                    <span style="color: #00a32a; font-size: 12px;">✅ Active</span>
                                </label>
                                <p style="margin: 8px 0 0 32px; color: #666; font-size: 13px;">Enhanced hook: gform_after_submission</p>
                            </div>
                            <?php endif; ?>
                            
                            <!-- Ninja Forms -->
                            <?php if (class_exists('Ninja_Forms')): ?>
                            <div style="padding: 15px; border: 2px solid <?php echo in_array('ninja_forms', $configured_plugins) ? '#25caf1' : '#ddd'; ?>; border-radius: 8px; background: <?php echo in_array('ninja_forms', $configured_plugins) ? '#f0fcff' : '#fff'; ?>;">
                                <label style="display: flex; align-items: center; gap: 10px;">
                                    <input type="checkbox" name="form_plugins[]" value="ninja_forms" <?php checked(in_array('ninja_forms', $configured_plugins)); ?> />
                                    <strong>Ninja Forms</strong>
                                    <span style="color: #00a32a; font-size: 12px;">✅ Active</span>
                                </label>
                                <p style="margin: 8px 0 0 32px; color: #666; font-size: 13px;">Enhanced hook: ninja_forms_after_submission</p>
                            </div>
                            <?php endif; ?>
                            
                            <!-- Fluent Forms -->
                            <?php if (class_exists('FluentForm\Framework\Foundation\Application')): ?>
                            <div style="padding: 15px; border: 2px solid <?php echo in_array('fluent_forms', $configured_plugins) ? '#25caf1' : '#ddd'; ?>; border-radius: 8px; background: <?php echo in_array('fluent_forms', $configured_plugins) ? '#f0fcff' : '#fff'; ?>;">
                                <label style="display: flex; align-items: center; gap: 10px;">
                                    <input type="checkbox" name="form_plugins[]" value="fluent_forms" <?php checked(in_array('fluent_forms', $configured_plugins)); ?> />
                                    <strong>Fluent Forms</strong>
                                    <span style="color: #00a32a; font-size: 12px;">✅ Active</span>
                                </label>
                                <p style="margin: 8px 0 0 32px; color: #666; font-size: 13px;">Enhanced hook: fluentform_submission_inserted</p>
                            </div>
                            <?php endif; ?>
                            
                            <!-- JotForm -->
                            <div style="padding: 15px; border: 2px solid <?php echo in_array('jotform', $configured_plugins) ? '#25caf1' : '#ddd'; ?>; border-radius: 8px; background: <?php echo in_array('jotform', $configured_plugins) ? '#f0fcff' : '#fff'; ?>;">
                                <label style="display: flex; align-items: center; gap: 10px;">
                                    <input type="checkbox" name="form_plugins[]" value="jotform" <?php checked(in_array('jotform', $configured_plugins)); ?> />
                                    <strong>JotForm (External)</strong>
                                    <span style="color: #2271b1; font-size: 12px;">🌐 Webhook Support</span>
                                </label>
                                <p style="margin: 8px 0 0 32px; color: #666; font-size: 13px;">Webhook URL: <?php echo home_url('?capitula_jotform_webhook=1'); ?></p>
                            </div>
                            
                        </div>
                        
                        <div style="text-align: center; margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd;">
                            <?php submit_button('Save Configuration', 'primary', 'submit_wizard', false, ['style' => 'font-size: 16px; padding: 10px 30px;']); ?>
                        </div>
                        
                    </form>
                </div>
                
                <!-- Status Sidebar -->
                <div style="flex: 1; background: #fff; padding: 20px; border: 1px solid #ccd0d4; box-shadow: 0 1px 1px rgba(0,0,0,.04); height: fit-content;">
                    <h3 style="margin-top: 0; color: #23282d;"><span class="dashicons dashicons-info"></span> Current Status</h3>
                    
                    <div style="margin-bottom: 15px; padding: 12px; background: <?php echo ($universal_enabled === 'yes') ? '#d1eddb' : '#fcf2cd'; ?>; border-left: 4px solid <?php echo ($universal_enabled === 'yes') ? '#00a32a' : '#dba617'; ?>;">
                        <strong>Universal Tracking:</strong><br>
                        <span style="color: <?php echo ($universal_enabled === 'yes') ? '#00a32a' : '#dba617'; ?>;">
                            <?php echo ($universal_enabled === 'yes') ? '✅ Enabled' : '⚠️ Disabled'; ?>
                        </span>
                    </div>
                    
                    <div style="margin-bottom: 15px; padding: 12px; background: #f0fcff; border-left: 4px solid #25caf1;">
                        <strong>CPL Method:</strong><br>
                        <span style="color: #25caf1;"><?php 
                            $method = get_option('capitula_cpl_tracking_method', 'simple_ajax');
                            echo $method === 'pure_php' ? '🛡️ Pure PHP' : '🚀 Simple AJAX';
                        ?></span>
                    </div>
                    
                    <?php if (!empty($configured_plugins)): ?>
                    <div style="margin-bottom: 15px; padding: 12px; background: #e8f5e8; border-left: 4px solid #4CAF50;">
                        <strong>Configured Plugins:</strong><br>
                        <ul style="margin: 8px 0 0 0; padding-left: 20px;">
                            <?php foreach ($configured_plugins as $plugin): ?>
                                <li style="color: #2e7d32; margin: 2px 0;"><?php 
                                    $names = [
                                        'contact_form_7' => 'Contact Form 7',
                                        'wpforms' => 'WPForms',
                                        'gravity_forms' => 'Gravity Forms',
                                        'ninja_forms' => 'Ninja Forms',
                                        'fluent_forms' => 'Fluent Forms',
                                        'jotform' => 'JotForm'
                                    ];
                                    echo $names[$plugin] ?? ucfirst(str_replace('_', ' ', $plugin));
                                ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    <?php endif; ?>
                    
                    <hr style="margin: 20px 0;">
                    
                    <h4 style="margin-bottom: 10px; color: #23282d;">Quick Actions:</h4>
                    <div style="display: flex; flex-direction: column; gap: 8px;">
                        <a href="<?php echo admin_url('admin.php?page=capitula-campaign-config'); ?>" class="button button-secondary" style="text-align: center;">
                            <span class="dashicons dashicons-admin-settings"></span> Main Settings
                        </a>
                        <a href="<?php echo admin_url('admin.php?page=capitula-status'); ?>" class="button button-secondary" style="text-align: center;">
                            <span class="dashicons dashicons-chart-line"></span> Debug Status
                        </a>
                    </div>
                    
                    <div style="margin-top: 20px; padding: 12px; background: #f6f7f7; border-radius: 4px; text-align: center; font-size: 12px; color: #666;">
                        <strong>Need Help?</strong><br>
                        Universal tracking works automatically<br>
                        with any form plugin!
                    </div>
                </div>
            </div>
        </div>
        
        <style>
        .capitula-wizard input[type="checkbox"]:checked + strong {
            color: #25caf1;
        }
        .capitula-wizard .form-plugins-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 15px;
        }
        </style>
        <?php
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        // Set default options (only if they don't already exist)
        add_option('capitula_enable_fallback', 'yes');  // Enable cross-domain fallback by default
        add_option('capitula_enable_woocommerce', 'yes');
        add_option('capitula_enable_contact_forms', 'yes');
        add_option('capitula_enable_registration', 'yes');
        
        // Use Pure PHP with Universal Tracking as default (new enhanced method)
        add_option('capitula_cpl_tracking_method', 'pure_php');
        
        // Universal tracking is compatible with CPA after interference fixes
        
        add_option('capitula_universal_tracking_enabled', 'yes');  // Enable universal tracking by default
        add_option('capitula_configured_form_plugins', array());  // Empty array - universal covers all
        
        // Send activation email notification
        $this->send_activation_notification();
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Send deactivation email notification
        $this->send_deactivation_notification();
        
        // Clean up transients
        global $wpdb;
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_capitula_%'");
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_capitula_%'");
    }
    
    /**
     * Send email notification when plugin is activated
     */
    private function send_activation_notification() {
        $site_info = $this->get_site_info();
        
        $subject = 'Capitula Plugin Activated - ' . $site_info['site_name'];
        
        $message = "<html><body style='font-family: Arial, sans-serif; line-height: 1.6; color: #333;'>";
        $message .= "<div style='max-width: 600px; margin: 0 auto; padding: 20px; border: 1px solid #ddd; border-radius: 5px;'>";
        $message .= "<h2 style='color: #25caf1; margin-top: 0;'>✅ Capitula Plugin Activated</h2>";
        $message .= "<p><strong>The Capitula Campaign Configuration plugin has been successfully activated.</strong></p>";
        $message .= "<div style='background: #f9f9f9; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
        $message .= "<h3 style='margin-top: 0; color: #333;'>Website Information:</h3>";
        $message .= "<ul style='list-style: none; padding: 0;'>";
        $message .= "<li><strong>Site Name:</strong> " . esc_html($site_info['site_name']) . "</li>";
        $message .= "<li><strong>Site URL:</strong> " . esc_html($site_info['site_url']) . "</li>";
        $message .= "<li><strong>Admin URL:</strong> " . esc_html($site_info['admin_url']) . "</li>";
        $message .= "<li><strong>WordPress Version:</strong> " . esc_html($site_info['wp_version']) . "</li>";
        $message .= "<li><strong>Plugin Version:</strong> " . esc_html($site_info['plugin_version']) . "</li>";
        $message .= "<li><strong>Activation Time:</strong> " . esc_html($site_info['timestamp']) . "</li>";
        if ($site_info['woocommerce_active']) {
            $message .= "<li><strong>WooCommerce:</strong> ✅ Active (Version: " . esc_html($site_info['woocommerce_version']) . ")</li>";
        } else {
            $message .= "<li><strong>WooCommerce:</strong> ❌ Not Active</li>";
        }
        $message .= "</ul>";
        $message .= "</div>";
        $message .= "<div style='background: #e8f5e8; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
        $message .= "<h3 style='margin-top: 0; color: #2e7d32;'>Active Features:</h3>";
        $message .= "<ul style='margin: 5px 0;'>";
        $message .= "<li>✅ CDN Integration</li>";
        $message .= "<li>✅ Campaign Token Handling</li>";
        if ($site_info['woocommerce_active']) {
            $message .= "<li>✅ WooCommerce Order Tracking (CPA)</li>";
        }
        $message .= "<li>✅ Universal Form Tracking (CPL)</li>";
        $message .= "<li>✅ User Registration Tracking</li>";
        $message .= "<li>✅ Cross-Domain Compatibility</li>";
        $message .= "</ul>";
        $message .= "</div>";
        $message .= "<p style='font-size: 12px; color: #666; margin-top: 30px; border-top: 1px solid #eee; padding-top: 15px;'>";
        $message .= "This is an automated notification from the Capitula Campaign Configuration plugin.<br>";
        $message .= "Server IP: " . esc_html($_SERVER['SERVER_ADDR'] ?? 'Unknown') . "<br>";
        $message .= "User Agent: " . esc_html($_SERVER['HTTP_USER_AGENT'] ?? 'Unknown');
        $message .= "</p>";
        $message .= "</div>";
        $message .= "</body></html>";
        
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: Capitula Plugin <no-reply@' . parse_url($site_info['site_url'], PHP_URL_HOST) . '>'
        );
        
        wp_mail('paul@capitula.co.uk', $subject, $message, $headers);
        
        // Log the activation
        error_log('Capitula Plugin: Activation notification sent to paul@capitula.co.uk for site: ' . $site_info['site_name']);
    }
    
    /**
     * Send email notification when plugin is deactivated
     */
    private function send_deactivation_notification() {
        $site_info = $this->get_site_info();
        
        $subject = 'Capitula Plugin Deactivated - ' . $site_info['site_name'];
        
        $message = "<html><body style='font-family: Arial, sans-serif; line-height: 1.6; color: #333;'>";
        $message .= "<div style='max-width: 600px; margin: 0 auto; padding: 20px; border: 1px solid #ddd; border-radius: 5px;'>";
        $message .= "<h2 style='color: #d32f2f; margin-top: 0;'>❌ Capitula Plugin Deactivated</h2>";
        $message .= "<p><strong>The Capitula Campaign Configuration plugin has been deactivated.</strong></p>";
        $message .= "<div style='background: #fff3cd; padding: 15px; border-radius: 5px; margin: 20px 0; border: 1px solid #ffeaa7;'>";
        $message .= "<p style='margin: 0; color: #856404;'><strong>⚠️ Important:</strong> All tracking functionality is now disabled on this website:</p>";
        $message .= "<ul style='color: #856404; margin: 10px 0 0 20px;'>";
        $message .= "<li>CDN scripts will not load</li>";
        $message .= "<li>Campaign tokens will not be processed</li>";
        $message .= "<li>Order conversion tracking (CPA) is disabled</li>";
        $message .= "<li>Form submission tracking (CPL) is disabled</li>";
        $message .= "<li>User registration tracking is disabled</li>";
        $message .= "</ul>";
        $message .= "</div>";
        $message .= "<div style='background: #f9f9f9; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
        $message .= "<h3 style='margin-top: 0; color: #333;'>Website Information:</h3>";
        $message .= "<ul style='list-style: none; padding: 0;'>";
        $message .= "<li><strong>Site Name:</strong> " . esc_html($site_info['site_name']) . "</li>";
        $message .= "<li><strong>Site URL:</strong> " . esc_html($site_info['site_url']) . "</li>";
        $message .= "<li><strong>Admin URL:</strong> " . esc_html($site_info['admin_url']) . "</li>";
        $message .= "<li><strong>WordPress Version:</strong> " . esc_html($site_info['wp_version']) . "</li>";
        $message .= "<li><strong>Plugin Version:</strong> " . esc_html($site_info['plugin_version']) . "</li>";
        $message .= "<li><strong>Deactivation Time:</strong> " . esc_html($site_info['timestamp']) . "</li>";
        $message .= "</ul>";
        $message .= "</div>";
        $message .= "<div style='background: #e3f2fd; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
        $message .= "<h3 style='margin-top: 0; color: #1565c0;'>To Reactivate:</h3>";
        $message .= "<ol style='margin: 5px 0;'>";
        $message .= "<li>Go to <a href='" . esc_url($site_info['admin_url'] . 'plugins.php') . "' style='color: #25caf1;'>WordPress Plugins</a></li>";
        $message .= "<li>Find \"Capitula Campaign Configuration\"</li>";
        $message .= "<li>Click \"Activate\"</li>";
        $message .= "</ol>";
        $message .= "</div>";
        $message .= "<p><strong>Need Support?</strong></p>";
        $message .= "<p>If the plugin was deactivated due to issues, please check:</p>";
        $message .= "<ul>";
        $message .= "<li>WordPress error logs</li>";
        $message .= "<li>Plugin conflict issues</li>";
        $message .= "<li>Server compatibility requirements</li>";
        $message .= "</ul>";
        $message .= "<p style='font-size: 12px; color: #666; margin-top: 30px; border-top: 1px solid #eee; padding-top: 15px;'>";
        $message .= "This is an automated notification from the Capitula Campaign Configuration plugin.<br>";
        $message .= "Server IP: " . esc_html($_SERVER['SERVER_ADDR'] ?? 'Unknown') . "<br>";
        $message .= "User Agent: " . esc_html($_SERVER['HTTP_USER_AGENT'] ?? 'Unknown');
        $message .= "</p>";
        $message .= "</div>";
        $message .= "</body></html>";
        
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: Capitula Plugin <no-reply@' . parse_url($site_info['site_url'], PHP_URL_HOST) . '>'
        );
        
        wp_mail('paul@capitula.co.uk', $subject, $message, $headers);
        
        // Log the deactivation
        error_log('Capitula Plugin: Deactivation notification sent to paul@capitula.co.uk for site: ' . $site_info['site_name']);
    }
    
    /**
     * Get site information for email notifications
     */
    private function get_site_info() {
        global $wp_version;
        
        $site_info = array(
            'site_name' => get_bloginfo('name'),
            'site_url' => home_url(),
            'admin_url' => admin_url(),
            'wp_version' => $wp_version,
            'plugin_version' => CAPITULA_PLUGIN_VERSION,
            'timestamp' => current_time('mysql'),
            'woocommerce_active' => class_exists('WooCommerce'),
            'woocommerce_version' => class_exists('WooCommerce') ? WC()->version : 'N/A'
        );
        
        return $site_info;
    }
}

// Initialize the plugin
CapitulaCampaignConfig::get_instance();

/**
 * Plugin Auto-Updater Class
 */
class CapitulaPluginUpdater {
    private $plugin_slug;
    private $version;
    private $update_server;
    private $cache_key;
    private $cache_allowed;
    private $github_data;
    
    public function __construct($plugin_slug, $version, $update_server) {
        $this->plugin_slug = $plugin_slug;
        $this->version = $version;
        $this->update_server = $update_server;
        $this->cache_key = 'capitula_plugin_update_' . md5($this->plugin_slug);
        $this->cache_allowed = true;
        
        add_filter('pre_set_site_transient_update_plugins', array($this, 'modify_transient'), 10, 1);
        add_filter('plugins_api', array($this, 'plugin_popup'), 10, 3);
        add_filter('upgrader_post_install', array($this, 'after_install'), 10, 3);
        
        // Force check for updates when needed
        add_action('admin_init', array($this, 'maybe_force_check'));
    }
    
    /**
     * Modify the plugin update transient to include our custom plugin
     */
    public function modify_transient($transient) {
        if (property_exists($transient, 'checked')) {
            if ($checked = $transient->checked) {
                $this->get_repository_info();
                
                if (isset($checked[$this->plugin_slug]) && isset($this->github_data['version'])) {
                    // Compare server version from API against installed version
                    $out_of_date = version_compare($this->github_data['version'], $this->version, '>');
                    
                    if ($out_of_date) {
                        $new_files = $this->get_new_version();
                        if ($new_files) {
                            $transient->response[$this->plugin_slug] = $new_files;
                        }
                    }
                }
            }
        }
        return $transient;
    }
    
    /**
     * Get plugin info from update server
     */
    private function get_repository_info() {
        if ($this->cache_allowed && $cache = get_transient($this->cache_key)) {
            $this->github_data = $cache;
        } else {
            $remote_version = wp_remote_get($this->update_server . '?action=get_version&slug=' . dirname($this->plugin_slug));
            
            if (!is_wp_error($remote_version)) {
                $this->github_data = json_decode(wp_remote_retrieve_body($remote_version), true);
                set_transient($this->cache_key, $this->github_data, 43200); // Cache for 12 hours
            }
        }
    }
    
    /**
     * Get new version info for WordPress
     */
    private function get_new_version() {
        $plugin_info = get_plugin_data(WP_PLUGIN_DIR . '/' . $this->plugin_slug);
        
        if ($this->github_data) {
            return (object) array(
                'slug' => dirname($this->plugin_slug),
                'plugin' => $this->plugin_slug,
                'new_version' => $this->github_data['version'],
                'url' => $this->github_data['homepage'],
                'package' => $this->github_data['download_url'],
                'requires' => $this->github_data['requires'],
                'tested' => $this->github_data['tested'],
                'requires_php' => $this->github_data['requires_php'],
            );
        }
        return false;
    }
    
    /**
     * Handle plugin info popup
     */
    public function plugin_popup($request, $action, $args) {
        if ($action !== 'plugin_information' || $args->slug !== dirname($this->plugin_slug)) {
            return $request;
        }
        
        $this->get_repository_info();
        
        if ($this->github_data) {
            return (object) array(
                'name' => $this->github_data['name'],
                'slug' => dirname($this->plugin_slug),
                'version' => $this->github_data['version'],
                'author' => $this->github_data['author'],
                'homepage' => $this->github_data['homepage'],
                'requires' => $this->github_data['requires'],
                'tested' => $this->github_data['tested'],
                'requires_php' => $this->github_data['requires_php'],
                'last_updated' => $this->github_data['last_updated'],
                'sections' => array(
                    'description' => $this->github_data['description'],
                    'changelog' => $this->github_data['changelog']
                ),
                'download_link' => $this->github_data['download_url']
            );
        }
        
        return $request;
    }
    
    /**
     * Handle post-install cleanup
     */
    public function after_install($response, $hook_extra, $result) {
        global $wp_filesystem;
        
        $install_directory = plugin_dir_path(__FILE__);
        $wp_filesystem->move($result['destination'], $install_directory);
        $result['destination'] = $install_directory;
        
        if ($this->cache_allowed) {
            delete_transient($this->cache_key);
        }
        
        return $result;
    }
    
    /**
     * Force check for updates if requested
     */
    public function maybe_force_check() {
        if (isset($_GET['capitula_force_update_check'])) {
            delete_transient($this->cache_key);
            wp_redirect(admin_url('plugins.php'));
            exit;
        }
    }
}

?>
