<?php
/**
 * Alternative CPL Tracking Approaches
 * These are completely different methods we could implement
 */

// OPTION 2: Pure PHP Server-Side Tracking (No JavaScript)
// ========================================================
// This approach handles everything on the server side when forms are submitted
// No JavaScript monitoring, no AJAX calls, just clean PHP processing

class CapitulaPurePHPTracking {
    
    /**
     * Contact Form 7 - Pure PHP approach
     */
    public function cf7_pure_php_tracking($contact_form) {
        $submission = WPCF7_Submission::get_instance();
        if ($submission) {
            $posted_data = $submission->get_posted_data();
            if (isset($posted_data['your-email'])) {
                $email = sanitize_email($posted_data['your-email']);
                if ($email) {
                    // Get campaign token from cookie
                    $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
                    if ($token) {
                        $this->send_cpl_tracking($email, $token);
                    }
                }
            }
        }
    }
    
    /**
     * Gravity Forms - Pure PHP approach
     */
    public function gf_pure_php_tracking($entry, $form) {
        foreach ($form['fields'] as $field) {
            if ($field->type === 'email' && !empty($entry[$field->id])) {
                $email = sanitize_email($entry[$field->id]);
                if ($email) {
                    // Get campaign token from cookie
                    $token = isset($_COOKIE['uac_token']) ? $_COOKIE['uac_token'] : '';
                    if ($token) {
                        $this->send_cpl_tracking($email, $token);
                    }
                    break;
                }
            }
        }
    }
    
    /**
     * Send CPL tracking directly from server
     */
    private function send_cpl_tracking($email, $token) {
        // Parse token
        $ad_position = substr($token, 0, 1);
        $campaign_token = substr($token, 1);
        
        // Get user IP
        $user_ip = $this->get_real_ip();
        
        // Get geolocation
        $geo_response = wp_remote_get('https://api.ipgeolocation.io/ipgeo?apiKey=d143802d9a6142b493df7085c11d16f6&ip=' . $user_ip);
        
        if (!is_wp_error($geo_response)) {
            $geo_data = json_decode(wp_remote_retrieve_body($geo_response), true);
            
            // Prepare CPL data
            $cpl_data = [
                'compaign_token' => $campaign_token,
                'user_email' => $email,
                'compaign_ip' => $geo_data['ip'] ?? $user_ip,
                'ad_position' => $ad_position,
                'device_type' => $this->get_device_type(),
                'compaign_country' => $geo_data['country_name'] ?? ''
            ];
            
            // Send to Capitula API
            wp_remote_post('https://demo.capitula.co.uk/external-api-cpl', [
                'headers' => ['Content-Type' => 'application/json'],
                'body' => json_encode($cpl_data),
                'timeout' => 30,
                'blocking' => false // Non-blocking so it doesn't slow down the form
            ]);
            
            // Log for debugging
            error_log('Capitula CPL sent for: ' . $email);
        }
    }
    
    private function get_real_ip() {
        $ip_keys = ['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR'];
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER)) {
                $ip = trim($_SERVER[$key]);
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        return '127.0.0.1';
    }
    
    private function get_device_type() {
        $user_agent = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');
        if (preg_match('/mobile|android|iphone|ipod/i', $user_agent)) {
            return 'Mobile';
        } elseif (preg_match('/tablet|ipad/i', $user_agent)) {
            return 'Tablet';
        }
        return 'Desktop';
    }
}

// OPTION 3: Webhook/API Endpoint Approach  
// =========================================
// Create custom WordPress REST API endpoints that forms can submit to

class CapitulaWebhookApproach {
    
    public function register_routes() {
        register_rest_route('capitula/v1', '/track-cpl', [
            'methods' => 'POST',
            'callback' => [$this, 'handle_cpl_webhook'],
            'permission_callback' => '__return_true',
            'args' => [
                'email' => ['required' => true, 'validate_callback' => 'is_email'],
                'token' => ['required' => true, 'sanitize_callback' => 'sanitize_text_field']
            ]
        ]);
    }
    
    public function handle_cpl_webhook($request) {
        $email = $request->get_param('email');
        $token = $request->get_param('token');
        
        // Process CPL tracking
        $result = $this->process_cpl($email, $token);
        
        return new WP_REST_Response($result, 200);
    }
    
    private function process_cpl($email, $token) {
        // Same CPL processing logic as above
        return ['success' => true, 'message' => 'CPL tracked'];
    }
}

// OPTION 4: Database Queue Approach
// ==================================
// Store CPL requests in database and process them via cron

class CapitulaDatabaseQueueApproach {
    
    public function store_cpl_for_processing($email, $token) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'capitula_cpl_queue';
        
        $wpdb->insert($table_name, [
            'email' => $email,
            'token' => $token,
            'user_ip' => $this->get_real_ip(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'],
            'created_at' => current_time('mysql'),
            'status' => 'pending'
        ]);
    }
    
    public function process_cpl_queue() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'capitula_cpl_queue';
        
        // Get pending CPL requests
        $pending_requests = $wpdb->get_results(
            "SELECT * FROM {$table_name} WHERE status = 'pending' LIMIT 10"
        );
        
        foreach ($pending_requests as $request) {
            // Process each CPL request
            $this->send_cpl_tracking($request);
            
            // Mark as processed
            $wpdb->update($table_name, 
                ['status' => 'processed'], 
                ['id' => $request->id]
            );
        }
    }
}

?>